//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeFilters/mdsMinMax.h           \n
 * Section: libVolume                           \n
 * Date:    2006/06/14                          \n
 *
 * $Id: mdsMinMax.hxx 50 2006-08-08 13:05:53Z spanel $
 * 
 * Description:
 * - Volume min and max filters.
 */


//==============================================================================
/*
 * Global functions.
 */

namespace minmax
{

//! Finds minimal value.
template <typename T>
T findMin(T *pData, tSize Size)
{
    MDS_CHECK(Size > 0, return T(0));

    T Min = pData[0];

    T *p = pData + 1;
    T *pMax = pData + Size;
    while( p < pMax )
    {
        if( *p < Min )
        {
            Min = *p;
        }
        ++p;
    }

    return Min;
}

//! Finds maximal value.
template <typename T>
T findMax(T *pData, tSize Size)
{
    MDS_CHECK(Size > 0, return T(0));

    T Max = pData[0];

    T *p = pData + 1;
    T *pMax = pData + Size;
    while( p < pMax )
    {
        if( *p > Max )
        {
            Max = *p;
        }
        ++p;
    }

    return Max;
}

} // namespace


//==============================================================================
/*
 * Methods templates.
 */

// Volume filtering method
template <class V, template <typename> class N>
bool CVolumeFilter<V, VF_MIN, N>::operator()(const V& SrcVolume, V& DstVolume)
{
    // Volume size
    tSize XCount = mds::math::getMin(SrcVolume.getXSize(), DstVolume.getXSize());
    tSize YCount = mds::math::getMin(SrcVolume.getYSize(), DstVolume.getYSize());
    tSize ZCount = mds::math::getMin(SrcVolume.getZSize(), DstVolume.getZSize());

    // Filter the image
    for( tSize z = 0; z < ZCount; ++z )
    {
        for( tSize y = 0; y < YCount; ++y )
        {
            for( tSize x = 0; x < XCount; ++x )
            {
                // Copy voxels from the window
                SrcVolume.copyWindow(x, y, z, m_MinSize, m_Data.getPtr());

                // Min finding
                tVoxel Min = minmax::findMin<tVoxel>(m_Data.getPtr(), m_Data.getSize());

                // Set pixel value
                DstVolume.set(x, y, z, Min);
            }
        }
    }

    // O.K.
    return true;
}


// Volume filter response
template <class V, template <typename> class N>
typename CVolumeFilter<V, VF_MIN, N>::tResult CVolumeFilter<V, VF_MIN, N>::getResponse(const V& SrcVolume, tSize x, tSize y, tSize z)
{
    // Copy voxels from the window
    SrcVolume.copyWindow(x, y, z, m_MinSize, m_Data.getPtr());

    // Min finding
    tVoxel Min = minmax::findMin<tVoxel>(m_Data.getPtr(), m_Data.getSize());
    return tResult(Min);
}


// Volume filtering method
template <class V, template <typename> class N>
bool CVolumeFilter<V, VF_MAX, N>::operator()(const V& SrcVolume, V& DstVolume)
{
    // Volume size
    tSize XCount = mds::math::getMin(SrcVolume.getXSize(), DstVolume.getXSize());
    tSize YCount = mds::math::getMin(SrcVolume.getYSize(), DstVolume.getYSize());
    tSize ZCount = mds::math::getMin(SrcVolume.getZSize(), DstVolume.getZSize());

    // Filter the image
    for( tSize z = 0; z < ZCount; ++z )
    {
        for( tSize y = 0; y < YCount; ++y )
        {
            for( tSize x = 0; x < XCount; ++x )
            {
                // Copy voxels from the window
                SrcVolume.copyWindow(x, y, z, m_MaxSize, m_Data.getPtr());

                // Max finding
                tVoxel Max = minmax::findMax<tVoxel>(m_Data.getPtr(), m_Data.getSize());

                // Set pixel value
                DstVolume.set(x, y, z, Max);
            }
        }
    }

    // O.K.
    return true;
}


// Volume filter response
template <class V, template <typename> class N>
typename CVolumeFilter<V, VF_MAX, N>::tResult CVolumeFilter<V, VF_MAX, N>::getResponse(const V& SrcVolume, tSize x, tSize y, tSize z)
{
    // Copy voxels from the window
    SrcVolume.copyWindow(x, y, z, m_MaxSize, m_Data.getPtr());

    // Max finding
    tVoxel Max = minmax::findMax<tVoxel>(m_Data.getPtr(), m_Data.getSize());
    return tResult(Max);
}

