//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeFilters/mdsMinMax.h           \n
 * Section: libVolume                           \n
 * Date:    2006/06/14                          \n
 *
 * $Id: mdsMinMax.h 50 2006-08-08 13:05:53Z spanel $
 * 
 * Description:
 * - Volume min and max filters.
 */

#ifndef MDS_VF_MINMAX_H
#define MDS_VF_MINMAX_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Math/mdsBase.h>
#include <MDSTk/Math/mdsVector.h>

#include "../mdsVolumeFilter.h"


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Min volume filter.
MDS_DECLARE_ID(VF_MIN);

//! Max volume filter.
MDS_DECLARE_ID(VF_MAX);


//==============================================================================
/*!
 * Template providing min-value volume filtering functions.
 * - Parameter V is a used volume type.
 */
template <class V, template <typename> class N>
class CVolumeFilter<V, VF_MIN, N> : public CVolumeFilterBase<V,N>
{
public:
    //! Volume filter base.
    typedef CVolumeFilterBase<V,N> base;
    typedef typename base::norm norm;
    typedef typename base::tVolume tVolume;
    typedef typename base::tVoxel tVoxel;
    typedef typename base::tResult tResult;

public:
    //! Constructor that creates a new filter.
    //! - Parameter 'Size' is a window size and it must be an odd number.
    CVolumeFilter(tSize Size)
            : m_Data(Size * Size * Size)
            , m_MinSize(Size)
    {
        MDS_ASSERT((Size % 2) == 1);
    }

    //! Destructor.
    ~CVolumeFilter() {}


    //! Min volume filtering.
    //! - Returns false on failure.
    bool operator()(const V& SrcVolume, V& DstVolume);

    //! Returns filter response at specified volume position.
    //! - Value is not normalized!
    tResult getResponse(const V& SrcVolume, tSize x, tSize y, tSize z);

    //! Sets the windows size.
    void setSize(tSize Size)
    {
        MDS_ASSERT((Size % 2) == 1);

        m_Data.resize(Size * Size * Size);
        m_MinSize = Size;
    }

protected:
    //! Vector used to find min value.
    mds::math::CVector<tVoxel> m_Data;

    //! Min filter size.
    tSize m_MinSize;
};


//==============================================================================
/*!
 * Template providing max-value volume filtering functions.
 * - Parameter V is a used volume type.
 */
template <class V, template <typename> class N>
class CVolumeFilter<V, VF_MAX, N> : public CVolumeFilterBase<V,N>
{
public:
    //! Volume filter base.
    typedef CVolumeFilterBase<V,N> base;
    typedef typename base::norm norm;
    typedef typename base::tVolume tVolume;
    typedef typename base::tVoxel tVoxel;
    typedef typename base::tResult tResult;

public:
    //! Constructor that creates a new filter.
    //! - Parameter 'Size' is a window size and it must be an odd number.
    CVolumeFilter(tSize Size)
            : m_Data(Size * Size * Size)
            , m_MaxSize(Size)
    {
        MDS_ASSERT((Size % 2) == 1);
    }

    //! Destructor.
    ~CVolumeFilter() {}


    //! Max volume filtering.
    //! - Returns false on failure.
    bool operator()(const V& SrcVolume, V& DstVolume);

    //! Returns filter response at specified volume position.
    //! - Value is not normalized!
    tResult getResponse(const V& SrcVolume, tSize x, tSize y, tSize z);

    //! Sets the windows size.
    void setSize(tSize Size)
    {
        MDS_ASSERT((Size % 2) == 1);

        m_Data.resize(Size * Size * Size);
        m_MaxSize = Size;
    }

protected:
    //! Vector used to find max value.
    mds::math::CVector<tVoxel> m_Data;

    //! Max filter size.
    tSize m_MaxSize;
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates.
#include "mdsMinMax.hxx"


} // namespace img
} // namespace mds

#endif // MDS_VF_MINMAX_H

