//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeFilters/mdsMedian.h           \n
 * Section: libVolume                           \n
 * Date:    2005/01/29                          \n
 *
 * $Id: mdsMedian.hxx 50 2006-08-08 13:05:53Z spanel $
 * 
 * Description:
 * - Volume median filter definition.
 */


//==============================================================================
/*
 * Global functions.
 */

namespace median
{

//! Median value finding (Z Algorithm)
template <typename T>
T findMedian(T *pData, tSize Size)
{
    // Index of the median value
    const tSize Median = Size / 2;

    // Z algorithm initialization
    tSize Left = 0;
    tSize Right = Size - 1;

    // Median finding
    while( Left < Right )
    {
        T Value = pData[Median];
        tSize LeftValue = Left;
        tSize RightValue = Right;

        while( RightValue >= Median && Median >= LeftValue )
        {
            while( pData[LeftValue] < Value )
            {
                ++LeftValue;
            }
            while( Value < pData[RightValue] )
            {
                --RightValue;
            }

            T Temp = pData[LeftValue];
            pData[LeftValue] = pData[RightValue];
            pData[RightValue] = Temp;

            ++LeftValue;
            --RightValue;
        }

        if( RightValue < Median )
        {
            Left = LeftValue;
        }
        if( Median < LeftValue )
        {
            Right = RightValue;
        }
    }

    return pData[Median];
}

} // namespace


//==============================================================================
/*
 * Methods templates.
 */

// Volume filtering method
template <class V, template <typename> class N>
bool CVolumeFilter<V, VF_MEDIAN, N>::operator()(const V& SrcVolume, V& DstVolume)
{
    // Volume size
    tSize XCount = mds::math::getMin(SrcVolume.getXSize(), DstVolume.getXSize());
    tSize YCount = mds::math::getMin(SrcVolume.getYSize(), DstVolume.getYSize());
    tSize ZCount = mds::math::getMin(SrcVolume.getZSize(), DstVolume.getZSize());

    // Filter the image
    for( tSize z = 0; z < ZCount; ++z )
    {
        for( tSize y = 0; y < YCount; ++y )
        {
            for( tSize x = 0; x < XCount; ++x )
            {
                // Copy voxels from the window
                SrcVolume.copyWindow(x, y, z, m_MedianSize, m_Data.getPtr());

                // Median finding
                tVoxel Median = median::findMedian<tVoxel>(m_Data.getPtr(), m_Data.getSize());

                // Set pixel value
                DstVolume.set(x, y, z, Median);
            }
        }
    }

    // O.K.
    return true;
}


// Volume filter response
template <class V, template <typename> class N>
typename CVolumeFilter<V, VF_MEDIAN, N>::tResult CVolumeFilter<V, VF_MEDIAN, N>::getResponse(const V& SrcVolume, tSize x, tSize y, tSize z)
{
    // Copy voxels from the window
    SrcVolume.copyWindow(x, y, z, m_MedianSize, m_Data.getPtr());

    // Median finding
    return tResult(median::findMedian<tVoxel>(m_Data.getPtr(), m_Data.getSize()));
}

