//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeFilters/mdsMedian.h           \n
 * Section: libVolume                           \n
 * Date:    2005/01/29                          \n
 *
 * $Id: mdsMedian.h 50 2006-08-08 13:05:53Z spanel $
 * 
 * Description:
 * - Volume median filter definition.
 */

#ifndef MDS_VF_MEDIAN_H
#define MDS_VF_MEDIAN_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Math/mdsBase.h>
#include <MDSTk/Math/mdsVector.h>

#include "../mdsVolumeFilter.h"

//#include <cstdlib>


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Median volume filter
MDS_DECLARE_ID(VF_MEDIAN);


//==============================================================================
/*!
 * Template providing median volume filtering functions.
 * - Parameter V is a used volume type.
 */
template <class V, template <typename> class N>
class CVolumeFilter<V, VF_MEDIAN, N> : public CVolumeFilterBase<V,N>
{
public:
    //! Volume filter base.
    typedef CVolumeFilterBase<V,N> base;
    typedef typename base::norm norm;
    typedef typename base::tVolume tVolume;
    typedef typename base::tVoxel tVoxel;
    typedef typename base::tResult tResult;

public:
    //! Constructor that creates a new median filter.
    //! - Parameter 'Size' is a window size and it must be an odd number.
    CVolumeFilter(tSize Size)
            : m_Data(Size * Size * Size)
            , m_MedianSize(Size)
    {
        MDS_ASSERT((Size % 2) == 1);
    }

    //! Destructor
    ~CVolumeFilter() {}


    //! Median volume filtering
    //! - Returns false on failure
    bool operator()(const V& SrcVolume, V& DstVolume);

    //! Returns filter response at specified volume position.
    //! - Value is not normalized!
    tResult getResponse(const V& SrcVolume, tSize x, tSize y, tSize z);

    //! Sets the windows size
    void setSize(tSize Size)
    {
        MDS_ASSERT((Size % 2) == 1);

        m_Data.resize(Size * Size * Size);
        m_MedianSize = Size;
    }

protected:
    //! Vector used to find median value.
    mds::math::CVector<tVoxel> m_Data;

    //! Median filter size
    tSize m_MedianSize;
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsMedian.hxx"


} // namespace img
} // namespace mds

#endif // MDS_VF_MEDIAN_H

