//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeFilters/mdsLaplacian.h        \n
 * Section: libVolume                           \n
 * Date:    2005/09/20                          \n
 *
 * $Id: mdsLaplacian.h 50 2006-08-08 13:05:53Z spanel $
 * 
 * Description:
 * - Laplacian volume filters.
 */

#ifndef MDS_VF_LAPLACIAN_H
#define MDS_VF_LAPLACIAN_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Math/mdsBase.h>

#include "../mdsVolumeFilter.h"
#include "../mdsVolumeFunctions.h"
#include "../mdsKernel.h"


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Laplacian volume filters
MDS_DECLARE_ID(VF_LAPLACIAN0);
MDS_DECLARE_ID(VF_LAPLACIAN1);


//==============================================================================
/*!
 * Laplacian volume filter (first version).
 * - Parameter V is a used volume type.
 * - Kernel definition: \n
 *    [ 0  0  0 ]   \n
 *    [ 0  1  0 ]   \n
 *    [ 0  0  0 ]   \n
 *   1 [ 0  1  0 ]  \n
 *   - [ 1 -6  1 ]  \n
 *   6 [ 0  1  0 ]  \n
 *      [ 0  0  0 ] \n
 *      [ 0  1  0 ] \n
 *      [ 0  0  0 ] \n
 */
template <class V, template <typename> class N>
class CVolumeFilter<V, VF_LAPLACIAN0, N> : public CVolumeFilterBase<V,N>
{
public:
    //! Volume filter base.
    typedef CVolumeFilterBase<V,N> base;
    typedef typename base::norm norm;
    typedef typename base::tVolume tVolume;
    typedef typename base::tVoxel tVoxel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response
    static const int DENOM = 6;

public:
    //! Default constructor.
    CVolumeFilter() {}

    //! Destructor.
    ~CVolumeFilter() {}

    //! Filtering of input/source volume.
    //! - Returns false on failure
    bool operator()(const V& SrcVolume, V& DstVolume);

    //! Returns filter response at specified volume position.
    //! - Value is not normalized!
    tResult getResponse(const V& SrcVolume, tSize x, tSize y, tSize z);

    //! Returns filter standard deviation.
    tSize getSize() const { return 3; }

};


//==============================================================================
/*!
 * Laplacian volume filter (second version).
 * - Parameter V is a used volume type.
 * - Kernel definition: \n
 *     [ 1  1  1 ]    \n
 *     [ 1  1  1 ]    \n
 *     [ 1  1  1 ]    \n
 *    1 [ 1   1  1 ]  \n
 *   -- [ 1 -26  1 ]  \n
 *   26 [ 1   1  1 ]  \n
 *        [ 1  1  1 ] \n
 *        [ 1  1  1 ] \n
 *        [ 1  1  1 ] \n
 */
template <class V, template <typename> class N>
class CVolumeFilter<V, VF_LAPLACIAN1, N> : public CVolumeFilterBase<V,N>
{
public:
    //! Volume filter base.
    typedef CVolumeFilterBase<V,N> base;
    typedef typename base::norm norm;
    typedef typename base::tVolume tVolume;
    typedef typename base::tVoxel tVoxel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response
    static const int DENOM = 26;

    //! Filter kernel
    static const CKernel3D::tData KERNEL[];

public:
    //! Default constructor.
    CVolumeFilter() {}

    //! Destructor.
    ~CVolumeFilter() {}


    //! Filtering of input/source volume.
    //! - Returns false on failure
    bool operator()(const V& SrcVolume, V& DstVolume);

    //! Returns filter response at specified volume position.
    //! - Value is not normalized!
    tResult getResponse(const V& SrcVolume, tSize x, tSize y, tSize z);

    //! Returns filter standard deviation.
    tSize getSize() const { return 3; }

};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsLaplacian.hxx"


} // namespace img
} // namespace mds

#endif // MDS_VF_LAPLACIAN_H

