//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeFilters/mdsGaussian.hxx       \n
 * Section: libVolume                           \n
 * Date:    2006/04/26                          \n
 *
 * $Id: mdsGaussian.hxx 50 2006-08-08 13:05:53Z spanel $
 * 
 * Description:
 * - 3D Gaussian smoothing filter.
 */


//==============================================================================
/*
 * Useful constants.
 */

namespace gaussian
{
//! Conversion of the filter standard deviation to the kernel size.
const double SIGMA2SIZE        = 5.0;

//! Conversion of the kernel size to the standard deviation.
const double SIZE2SIGMA        = 1 / 5.0;
}


//==============================================================================
/*
 * Methods templates.
 */

template <class V, template <typename> class N>
inline void CVolumeFilter<V, VF_GAUSSIAN, N>::resize(tSize Size)
{
    CKernel2D::resize(Size);

    initKernel(size2Sigma(Size));
}


template <class V, template <typename> class N>
inline void CVolumeFilter<V, VF_GAUSSIAN, N>::setSigma(double dSigma)
{
    MDS_ASSERT(dSigma > 0.0);

    CKernel2D::resize(sigma2Size(dSigma));

    initKernel(dSigma);
}


template <class V, template <typename> class N>
inline tSize CVolumeFilter<V, VF_GAUSSIAN, N>::sigma2Size(double dSigma)
{
    tSize Size = tSize(gaussian::SIGMA2SIZE * dSigma) + 1;
    if( (Size % 2) == 1 )
    {
        return Size;
    }
    else
    {
        return Size + 1;
    }
}


template <class V, template <typename> class N>
inline double CVolumeFilter<V, VF_GAUSSIAN, N>::size2Sigma(tSize Size)
{
    return gaussian::SIZE2SIGMA * (Size - 1);
}


template <class V, template <typename> class N>
inline double CVolumeFilter<V, VF_GAUSSIAN, N>::getGaussianFuncValue(double dX,
                                                                     double dY,
                                                                     double dZ,
                                                                     double dSigma
                                                                     )
{
    static const double dConst = 1.0 / pow(mds::math::TWO_PI, 1.5);

    double dS3 = 1.0 / (dSigma * dSigma * dSigma);
    double dS2 = dS2 * dSigma;

    return dConst * dS3 * exp(-0.5 * dS2 * (dX * dX + dY * dY + dZ * dZ));
}


template <class V, template <typename> class N>
void CVolumeFilter<V, VF_GAUSSIAN, N>::initKernel(double dSigma)
{
    static const double dConst = 1.0 / pow(mds::math::TWO_PI, 1.5);

    // Set the standard deviation
    m_dSigma = dSigma;

    // Helper values
    double dS3 = 1.0 / (dSigma * dSigma * dSigma);
    double dS2 = dS3 * dSigma;
    double dA = dConst * dS3;
    double dB = -0.5 * dS2;

    // Half of the kernel size
    tSize HalfSize = getSize() / 2;

    // Initialize the filter kernel
    for( tSize k = -HalfSize; k <= HalfSize; ++k )
    {
        for( tSize j = -HalfSize; j <= HalfSize; ++j )
        {
            for( tSize i = -HalfSize; i <= HalfSize; ++i )
            {
                double dValue = dA * exp(dB * (i * i + j * j + k * k));
                set(i + HalfSize, j + HalfSize, k + HalfSize, tData(dValue));
            }
        }
    }
}


// Volume filtering method
template <class V, template <typename> class N>
bool CVolumeFilter<V, VF_GAUSSIAN, N>::operator()(const V& SrcVolume, V& DstVolume)
{
    // Volume size
    tSize XCount = mds::math::getMin(SrcVolume.getXSize(), DstVolume.getXSize());
    tSize YCount = mds::math::getMin(SrcVolume.getYSize(), DstVolume.getYSize());
    tSize ZCount = mds::math::getMin(SrcVolume.getZSize(), DstVolume.getZSize());

    // Filter the image
    for( tSize z = 0; z < ZCount; ++z )
    {
        for( tSize y = 0; y < YCount; ++y )
        {
            for( tSize x = 0; x < XCount; ++x )
            {
                tResult Value = getResponse(SrcVolume, x, y, z);
                DstVolume.set(x, y, z, norm::normalize(Value));
            }
        }
    }

    // O.K.
    return true;
}


//==============================================================================
/*
 * Methods templates.
 */

// Filter kernel
template <class V, template <typename> class N>
const CKernel3D::tData CVolumeFilter<V, VF_GAUSSIAN_3, N>::KERNEL[] =
{
    0.0f,  1.0f,  0.0f,
    1.0f,  3.0f,  1.0f,
    0.0f,  1.0f,  0.0f,

    1.0f,  3.0f,  1.0f,
    3.0f,  9.0f,  3.0f,
    1.0f,  3.0f,  1.0f,

    0.0f,  1.0f,  0.0f,
    1.0f,  3.0f,  1.0f,
    0.0f,  1.0f,  0.0f
};


// Volume filtering method
template <class V, template <typename> class N>
bool CVolumeFilter<V, VF_GAUSSIAN_3, N>::operator()(const V& SrcVolume, V& DstVolume)
{
    // Volume size
    tSize XCount = mds::math::getMin(SrcVolume.getXSize(), DstVolume.getXSize());
    tSize YCount = mds::math::getMin(SrcVolume.getYSize(), DstVolume.getYSize());
    tSize ZCount = mds::math::getMin(SrcVolume.getZSize(), DstVolume.getZSize());

    // Filter the volume
    for( tSize z = 0; z < ZCount; ++z )
    {
        for( tSize y = 0; y < YCount; ++y )
        {
            for( tSize x = 0; x < XCount; ++x )
            {
                tResult Value = getResponse(SrcVolume, x, y, z);
                DstVolume.set(x, y, z, norm::normalize(Value));
            }
        }
    }

    // O.K.
    return true;
}


// Volume filter response
template <class V, template <typename> class N>
typename CVolumeFilter<V, VF_GAUSSIAN_3, N>::tResult CVolumeFilter<V, VF_GAUSSIAN_3, N>::getResponse(const V& SrcVolume, tSize x, tSize y, tSize z)
{
    // Output normalization
    static const tResult Denom = 1.0f / DENOM;

    // Filter kernel
    static const CKernel3D Kernel(KERNEL, getSize());

    // Compute filter response
    return Denom * convolve<tResult>(SrcVolume, x, y, z, Kernel);
}

