//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeFilters/mdsConvolution.h      \n
 * Section: libImage                            \n
 * Date:    2004/05/04                          \n
 *
 * $Id: mdsConvolution.h 442 2007-08-10 15:22:27Z spanel $
 *
 * Description:
 * - Convolution volume filter.
 */

#ifndef MDS_VF_CONVOLUTION_H
#define MDS_VF_CONVOLUTION_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Math/mdsBase.h>

#include "../mdsVolumeFilter.h"
#include "../mdsVolumeFunctions.h"
#include "../mdsKernel.h"


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Basic convolution volume filter
MDS_DECLARE_ID(VF_CONVOLUTION);


//==============================================================================
/*!
 * General convolution volume filter.
 * - Parameter V is a used volume type.
 */
template <class V, template <typename> class N>
class CVolumeFilter<V, VF_CONVOLUTION, N>
    : public CVolumeFilterBase<V,N>
    , public CKernel3D
{
public:
    //! Volume filter base.
    typedef CVolumeFilterBase<V,N> base;
    typedef typename base::norm norm;
    typedef typename base::tVolume tVolume;
    typedef typename base::tVoxel tVoxel;
    typedef typename base::tResult tResult;

public:
    //! Constructor that creates a new 2D convolution kernel
    //! - Parameter 'Size' must be an odd number
    CVolumeFilter(tSize Size) : CKernel3D(Size) {}

    //! Constructor
    CVolumeFilter(tSize Size, const tData& Value)
        : CKernel3D(Size, Value)
    {}

    //! Constructor
    CVolumeFilter(const tConstDataPtr pData, tSize Size)
        : CKernel3D(pData, Size)
    {}

    //! Constructor
    //! - Filter is based on a given convolution kernel
    CVolumeFilter(const CKernel3D& k) : CKernel3D(k) {}

    //! Destructor
    ~CVolumeFilter() {}


    //! Filtering of input/source volume. Point filter responses are written
    //! to the destination volume.
    //! - Returns false on failure
    bool operator()(const V& SrcVolume, V& DstVolume);

    //! Returns filter response at specified volume position.
    //! - Value is not normalized!
    tResult getResponse(const V& SrcVolume, tSize x, tSize y, tSize z)
    {
        // Compute filter response
        return convolve<tResult>(SrcVolume, x, y, z, *this);
    }
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates.
#include "mdsConvolution.hxx"


} // namespace img
} // namespace mds

#endif // MDS_VOLUMEFILTER_H

