//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeFilters/mdsAnisotropic.h      \n
 * Section: libImage                            \n
 * Date:    2007/05/03                          \n
 *
 * $Id:$
 *
 * Description:
 * - Anisotropic filtering based on the diffusion process.
 */


//==============================================================================
/*
 * Methods templates.
 */

// Integration constant chosen according to the number of neighbours
template <class V, template <typename> class N>
const double CVolumeFilter<V, VF_ANISOTROPIC, N>::DT          = 0.166666666666667;

// Minimal allowed change of the 
template <class V, template <typename> class N>
const double CVolumeFilter<V, VF_ANISOTROPIC, N>::MIN_CHANGE  = 0.1;


template <class V, template <typename> class N>
inline double CVolumeFilter<V, VF_ANISOTROPIC, N>::getDiffusionStrength(double dGrad)
{
    // Magnitude of the image gradient
    double dMagnitude = mds::math::getAbs(dGrad);
    
    // Evaluate the diffusion function
    double dTemp = m_dInvKappa * dMagnitude;
    return std::exp(-dTemp * dTemp);
}


template <class V, template <typename> class N>
inline typename CVolumeFilter<V, VF_ANISOTROPIC, N>::tResult CVolumeFilter<V, VF_ANISOTROPIC, N>::getResponse(const V& SrcVolume, tSize x, tSize y, tSize z)
{
    MDS_THROW_ERROR("CVolumeFilter<V,VF_ANISOTROPIC,N>::getResponse(): Function not implemented");

    return tResult(0);
}


// Image filtering method
template <class V, template <typename> class N>
bool CVolumeFilter<V, VF_ANISOTROPIC, N>::operator()(const V& SrcVolume, V& DstVolume)
{   
    // Volume size
    tSize XSize = mds::math::getMin(SrcVolume.getXSize(), DstVolume.getXSize());
    tSize YSize = mds::math::getMin(SrcVolume.getYSize(), DstVolume.getYSize());
    tSize ZSize = mds::math::getMin(SrcVolume.getZSize(), DstVolume.getZSize());
    tSize Margin = SrcVolume.getMargin();

    // Check the margin
    MDS_CHECK(Margin >= 1, return false);

    // Helper volume
/*    CFVolume FlowVolume(XSize, YSize, BLOCK_SIZE);

    // Copy the source volume
    DstVolume = SrcVolume;

    // Process the input volume per blocks
    for( tSize k = 0; ZSize > 0; ZSize -= BLOCK_SIZE, k += BLOCK_SIZE )
    {
        tSize sz = mds::math::getMin<tSize>(BLOCK_SIZE, ZSize);
        
        // Create block view of the original and destination volume
        tVolume SrcVolumeRef(SrcVolume, 0, 0, k, XSize, YSize, sz, mds::REFERENCE);
        tVolume DstVolumeRef(DstVolume, 0, 0, k, XSize, YSize, sz, mds::REFERENCE);
        
        // Initial value of the flow
        double dFlow = 1.0;
        
        // Diffusion process
        tSize x, y, z;
        for( tSize iter = 0; ; ++iter )
        {
            // Check the number of iterations
            if( m_NumOfIters > 0 && iter >= m_NumOfIters )
            {
                break;
            }
            
            // Clear the flow volume
            FlowVolume.fill(0);
            
            // Filter the volume
            for( z = 0; z < BLOCK_SIZE; ++z )
            {
                for( y = 0; y < YSize; ++y )
                {
                    for( x = 0; x < XSize; ++x )
                    {
                        // Evaluate volume derivatives
                        double dGradDown = double(DstVolumeRef(x,y+1,z)) - double(DstVolumeRef(x,y,z));
                        double dGradTop = double(DstVolumeRef(x,y,z)) - double(DstVolumeRef(x,y-1,z));
                        double dGradRight = double(DstVolumeRef(x+1,y,z)) - double(DstVolumeRef(x,y,z));
                        double dGradLeft = double(DstVolumeRef(x,y,z)) - double(DstVolumeRef(x-1,y,z));
                        double dGradFront = double(DstVolumeRef(x,y,z+1)) - double(DstVolumeRef(x,y,z));
                        double dGradBack = double(DstVolumeRef(x,y,z)) - double(DstVolumeRef(x,y,z-1));
    
                        // Evaluate the diffusion function
                        double dFlowDown = getDiffusionStrength(dGradDown) * dGradDown;
                        double dFlowTop = getDiffusionStrength(dGradTop) * dGradTop;
                        double dFlowRight = getDiffusionStrength(dGradRight) * dGradRight;
                        double dFlowLeft = getDiffusionStrength(dGradLeft) * dGradLeft;
                        double dFlowFront = getDiffusionStrength(dGradFront) * dGradFront;
                        double dFlowBack = getDiffusionStrength(dGradBack) * dGradBack;     
        
                        // Calculate the flow
                        double dDiff = double(SrcVolumeRef(x,y,z)) - double(DstVolumeRef(x,y,z));
                        FlowVolume(x,y,z) = tFloatPixel(DT * (dFlowRight - dFlowLeft + dFlowDown - dFlowTop + dFlowFront - dFlowBack + dDiff));
                    }
                }
            }
            
            // Modify the volume and calculate the convergence criterion
            double dNewFlow = 0.0;
            for( z = 0; z < BLOCK_SIZE; ++z )
            {
                for( y = 0; y < YSize; ++y )
                {
                    for( x = 0; x < XSize; ++x )
                    {
                        DstVolumeRef(x,y,z) = tVoxel(tFloatPixel(DstVolumeRef(x,y,z)) + FlowVolume(x,y,z));
                        dNewFlow += mds::math::getAbs(FlowVolume(x,y,z));
                    }
                }
            }
            
            // Estimate change of the volume
            double dDelta = mds::math::getAbs(dNewFlow / dFlow - 1.0);
            if( dDelta < MIN_CHANGE )
            {
                break;
            }
            dFlow = dNewFlow;
        }
    }*/

    // Helper volumes
/*    CFVolume FlowVolume(XSize, YSize, ZSize);

    // Copy the source volume
    DstVolume = SrcVolume;

    // Initial value of the flow
    double dFlow = 1.0;

    // Diffusion process
    tSize x, y, z;
    for( tSize iter = 0; ; ++iter )
    {
        // Check the number of iterations
        if( m_NumOfIters > 0 && iter >= m_NumOfIters )
        {
            break;
        }
        
        // Clear the flow volume
        FlowVolume.fill(0);
        
        // Filter the volume
        for( z = 0; z < ZSize; ++z )
        {
            for( y = 0; y < YSize; ++y )
            {
                for( x = 0; x < XSize; ++x )
                {
                    // Evaluate volume derivatives
                    double dGradDown = double(DstVolume(x,y+1,z)) - double(DstVolume(x,y,z));
                    double dGradTop = double(DstVolume(x,y,z)) - double(DstVolume(x,y-1,z));
                    double dGradRight = double(DstVolume(x+1,y,z)) - double(DstVolume(x,y,z));
                    double dGradLeft = double(DstVolume(x,y,z)) - double(DstVolume(x-1,y,z));
                    double dGradFront = double(DstVolume(x,y,z+1)) - double(DstVolume(x,y,z));
                    double dGradBack = double(DstVolume(x,y,z)) - double(DstVolume(x,y,z-1));

                    // Evaluate the diffusion function
                    double dFlowDown = getDiffusionStrength(dGradDown) * dGradDown;
                    double dFlowTop = getDiffusionStrength(dGradTop) * dGradTop;
                    double dFlowRight = getDiffusionStrength(dGradRight) * dGradRight;
                    double dFlowLeft = getDiffusionStrength(dGradLeft) * dGradLeft;
                    double dFlowFront = getDiffusionStrength(dGradFront) * dGradFront;
                    double dFlowBack = getDiffusionStrength(dGradBack) * dGradBack;     
    
                    // Calculate the flow
                    double dDiff = double(SrcVolume(x,y,z)) - double(DstVolume(x,y,z));
                    FlowVolume(x,y,z) = tFloatPixel(DT * (dFlowRight - dFlowLeft + dFlowDown - dFlowTop + dFlowFront - dFlowBack + dDiff));
                }
            }
        }
        
        // Modify the volume and calculate the convergence criterion
        double dNewFlow = 0.0;
        for( z = 0; z < ZSize; ++z )
        {
            for( y = 0; y < YSize; ++y )
            {
                for( x = 0; x < XSize; ++x )
                {
                    DstVolume(x,y,z) = tVoxel(tFloatPixel(DstVolume(x,y,z)) + FlowVolume(x,y,z));
                    dNewFlow += mds::math::getAbs(FlowVolume(x,y,z));
                }
            }
        }

        // Estimate change of the volume
        double dDelta = mds::math::getAbs(dNewFlow / dFlow - 1.0);
        if( dDelta < MIN_CHANGE )
        {
            break;
        }
        dFlow = dNewFlow;
    }*/

    // Copy the source volume
    DstVolume = SrcVolume;

    // Initial value of the flow
    double dFlow = 1.0;

    // Diffusion process
    tSize x, y, z;
    for( tSize iter = 0; ; ++iter )
    {
        // Check the number of iterations
        if( m_NumOfIters > 0 && iter >= m_NumOfIters )
        {
            break;
        }
               
        // Filter the volume
        double dNewFlow = 0.0;
        for( z = 0; z < ZSize; ++z )
        {
            for( y = 0; y < YSize; ++y )
            {
                for( x = 0; x < XSize; ++x )
                {
                    // Evaluate volume derivatives
                    double dGradDown = double(DstVolume(x,y+1,z)) - double(DstVolume(x,y,z));
                    double dGradTop = double(DstVolume(x,y,z)) - double(DstVolume(x,y-1,z));
                    double dGradRight = double(DstVolume(x+1,y,z)) - double(DstVolume(x,y,z));
                    double dGradLeft = double(DstVolume(x,y,z)) - double(DstVolume(x-1,y,z));
                    double dGradFront = double(DstVolume(x,y,z+1)) - double(DstVolume(x,y,z));
                    double dGradBack = double(DstVolume(x,y,z)) - double(DstVolume(x,y,z-1));

                    // Evaluate the diffusion function
                    double dFlowDown = getDiffusionStrength(dGradDown) * dGradDown;
                    double dFlowTop = getDiffusionStrength(dGradTop) * dGradTop;
                    double dFlowRight = getDiffusionStrength(dGradRight) * dGradRight;
                    double dFlowLeft = getDiffusionStrength(dGradLeft) * dGradLeft;
                    double dFlowFront = getDiffusionStrength(dGradFront) * dGradFront;
                    double dFlowBack = getDiffusionStrength(dGradBack) * dGradBack;     

                    // Calculate the flow
                    double dDiff = double(SrcVolume(x,y,z)) - double(DstVolume(x,y,z));
                    double dDelta = DT * (dFlowRight - dFlowLeft + dFlowDown - dFlowTop + dFlowFront - dFlowBack + dDiff);
                    DstVolume(x,y,z) = tVoxel(double(DstVolume(x,y,z)) + dDelta);
                    dNewFlow += mds::math::getAbs(dDelta);
                }
            }
        }
        
        // Estimate change of the volume
        double dDelta = mds::math::getAbs(dNewFlow / dFlow - 1.0);
        if( dDelta < MIN_CHANGE )
        {
            break;
        }
        dFlow = dNewFlow;
    }

    // O.K.
    return true;
}

