//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeEdgeDetectors/mdsCanny.h      \n
 * Section: libImage                            \n
 * Date:    2006/04/24                          \n
 *
 * $Id: mdsCanny.h 336 2007-05-04 18:11:37Z spanel $
 *
 * Description:
 * - Canny edge detector in 3D.
 */

#ifndef MDS_VED_CANNY_H
#define MDS_VED_CANNY_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Math/mdsBase.h>

#include "../mdsPixelTraits.h"
#include "../mdsVolumeEdgeDetector.h"
#include "../mdsVolumeFilters.h"
#include "../mdsVector.h"

#include <cmath>


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Canny edge detector in 3D.
MDS_DECLARE_ID(VED_CANNY);


//==============================================================================
/*!
 * Canny edge detector in 3D space.
 * - Parameter V is a used volume type.
 */
template <class V>
class CVolumeEdgeDetector<V, VED_CANNY> : public CVolumeEdgeDetectorBase<V>
{
public:
    //! Volume filter base.
    typedef CVolumeEdgeDetectorBase<V> base;
    typedef typename base::tVolume tVolume;
    typedef typename base::tVoxel tVoxel;
    typedef typename base::tResult tResult;
    typedef typename base::tVolumeIterator tVolumeIterator;

    //! Size of the processing block (volume window). The input volume
    //! is divided into smaller blocks - cubes for the processing.
    //! - This is used to reduce amount of the memory required by the
    //!   filtering.
    enum { BLOCK_SIZE = 64 };

public:
    //! Default constructor.
    //! - Thresholds for the hysteresis (0..1).
    CVolumeEdgeDetector(double dT1, double dT2)
        : m_dT1(dT1)
        , m_dT2(dT2)
    {
        MDS_ASSERT(dT1 <= 1.0 && dT2 <= dT1 && dT2 >= 0.0);
    }

    //! Destructor.
    ~CVolumeEdgeDetector() {}


    //! Edge detection.
    //! - Magnitude volume is produced.
    //! - Returns false on failure.
    bool operator()(const V& SrcVolume, V& DstVolume);

    //! Returns thresholds of the non-maximal suppression algorithm.
    inline void getThresholds(double& dT1, double& dT2);

    //! Sets thresholds of the non-maximal suppression algorithm.
    inline void setThresholds(double dT1, double dT2);

protected:
    //! Gaussian smoothing filter.
    CVolumeFilter<tVolume, VF_GAUSSIAN_3> m_GaussFilter;

    //! Sobel operators.
    CVolumeFilter<tVolume, VF_SOBEL_X, VFN_MEAN> m_SobelX;
    CVolumeFilter<tVolume, VF_SOBEL_Y, VFN_MEAN> m_SobelY;
    CVolumeFilter<tVolume, VF_SOBEL_Z, VFN_MEAN> m_SobelZ;

    //! Thresholds used by hysteresis algorithm.
    double m_dT1, m_dT2;

protected:
    //! Non-maximal suppression.
    void nonMaxSuppression(const tVolume& GradVolumeX,
                           const tVolume& GradVolumeY,
                           const tVolume& GradVolumeZ,
                           tVolume& MagnitudeVolume,
                           tVolume& Volume,
                           tSize x,
                           tSize y,
                           tSize z,
                           tSize XSize,
                           tSize YSize,
                           tSize ZSize
                           );

    //! Hystersis controlled by two given thresholds.
    //! - Applies the hysteresis to the provided volume. It checks each voxel
    //!   against an upper threshhold 'T1'. If the voxel value is equal
    //!   or higher, the current voxel is an edge voxel. Then all direct
    //!   neighbours of the voxel are compared to a lower threshhold 'T2'
    //!   and if their values are equal or higher than this threshhold,
    //!   they become edge voxels as well.
    //! - In practice the threshold T1 = (2-3) * T2.
    void hysteresis(tVolume& MagnitudeVolume,
                    tVolume& Volume,
                    tSize x,
                    tSize y,
                    tSize z,
                    tSize XSize,
                    tSize YSize,
                    tSize ZSize,
                    tVoxel T1,
                    tVoxel T2
                    );

    //! Returns true if any neighbouring voxel has greater value
    //! than a given threshold.
    bool checkNeighbours(tVolume& Volume,
                         tSize x,
                         tSize y,
                         tSize z,
                         tVoxel T
                         );
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsCanny.hxx"


} // namespace img
} // namespace mds

#endif // MDS_VED_CANNY_H

