//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeCornerDetectors/mdsSusan.h    \n
 * Section: libImage                            \n
 * Date:    2007/07/03                          \n
 *
 * $Id:$
 *
 * Description:
 * - Volume 3D corner/landmark detector based on Susan image corner detector.
 */

#ifndef MDS_VCD_SUSAN_H
#define MDS_VCD_SUSAN_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Math/mdsBase.h>
#include <MDSTk/Math/mdsStaticMatrix.h>
#include <MDSTk/Math/mdsMatrixFunctions.h>

#include "../mdsPixelTraits.h"
#include "../mdsVolumeCornerDetector.h"
#include "../mdsVolumeFilters.h"

#include <cmath>


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Volume 3D corner/landmark detector based on Susan image corner detector.
MDS_DECLARE_ID(VCD_SUSAN);


//==============================================================================
/*!
 * - Volume 3D corner/landmark detector based on Susan image corner detector.
 * - Parameter V is a used volume type.
 */
template <class V>
class CVolumeCornerDetector<V, VCD_SUSAN> : public CVolumeCornerDetectorBase<V>
{
public:
    //! Volume filter base.
    typedef CVolumeCornerDetectorBase<V> base;
    typedef typename base::tVolume tVolume;
    typedef typename base::tVoxel tVoxel;
    typedef typename base::tResult tResult;
    typedef typename base::tVolumeIterator tVolumeIterator;
    typedef typename base::tCorners tCorners;

    //! Size of the processing block (volume window). The input volume
    //! is divided into smaller blocks - cubes for the processing.
    //! - This is used to reduce amount of the memory required by the
    //!   filtering.
    enum { BLOCK_SIZE = 64 };

    //! Maximal value stored in the internal lookup table.
    enum { MAX_LUT = 255 };

    //! Size of the internal lookup table.
    enum { LUT_SIZE = 2 * MAX_LUT + 1 };

    //! Threshold used to suppress non-corner points.
    enum { DIFF_THRESHOLD = 700 };

public:
    //! Default constructor.
    //! - Detection threshold.
    //! - Third parameter affects evaluation of difference of two voxels.
    inline CVolumeCornerDetector(double dThreshold, double dWidth = 20.0);

    //! Destructor.
    ~CVolumeCornerDetector() {}


    //! Volume landmark detection.
    //! - Returns false on failure.
    bool operator()(const V& SrcVolume, tCorners& Corners);

    //! Returns cornerness threshold.
    double getThreshold() { return m_dThreshold; }

    //! Sets cornerness threshold.
    inline void setThreshold(double dThreshold);

    //! Returns the width parameter.
    double getWidth() { return m_dWidth; }

    //! Sets the width parameter.
    inline void setWidth(double dWidth);

protected:
    //! Corner detector parameters.
    double m_dThreshold, m_dWidth, m_dInvWidth;

    //! Internal lookup table used for fast computation of the exp() function.
    mds::math::CStaticVector<int, LUT_SIZE> m_LUT;

protected:
    //! Returns true if any neighbouring voxel has greater value
    //! than a given threshold.
    inline bool checkNeighbours(CFVolume& Volume,
                                tSize x,
                                tSize y,
                                tSize z,
                                tFloatPixel T
                                );

    //! Initializes internal lookup table.
    void initLUT();

    //! Returns difference of two given pixels.
    inline int getDiff(tPixel8 a, tPixel8 b);
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsSusan.hxx"


} // namespace img
} // namespace mds

#endif // MDS_VCD_SUSAN_H

