//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    VolumeCornerDetectors/mdsRohr.h     \n
 * Section: libImage                            \n
 * Date:    2007/07/03                          \n
 *
 * $Id:$
 *
 * Description:
 * - Volume data corner/landmark detector based on Rohr's 3D differential operators.
 * - Algorithm is partially similar to the well-known Harris image corner detector.
 */

#ifndef MDS_VCD_ROHR_H
#define MDS_VCD_ROHR_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Math/mdsBase.h>
#include <MDSTk/Math/mdsStaticMatrix.h>
#include <MDSTk/Math/mdsMatrixFunctions.h>

#include "../mdsPixelTraits.h"
#include "../mdsVolumeCornerDetector.h"
#include "../mdsVolumeFilters.h"

#include <cmath>


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Modified volume corner/landmark detector based on Rhor's 3D differential operators.
MDS_DECLARE_ID(VCD_ROHR);


//==============================================================================
/*!
 * - Volume 3D corner/landmark detector based on Rhor's 3D differential operators.
 * - Algorithm is partially similar to the well-known Harris image corner detector.
 * - Parameter V is a used volume type.
 */
template <class V>
class CVolumeCornerDetector<V, VCD_ROHR> : public CVolumeCornerDetectorBase<V>
{
public:
    //! Volume filter base.
    typedef CVolumeCornerDetectorBase<V> base;
    typedef typename base::tVolume tVolume;
    typedef typename base::tVoxel tVoxel;
    typedef typename base::tResult tResult;
    typedef typename base::tVolumeIterator tVolumeIterator;
    typedef typename base::tCorners tCorners;

    //! Size of the processing block (volume window). The input volume
    //! is divided into smaller blocks - cubes for the processing.
    //! - This is used to reduce amount of the memory required by the
    //!   filtering.
    enum { BLOCK_SIZE = 64 };

public:
    //! Default constructor.
    //! - Size of the window (must be an odd number) used to estimate eigen values.
    //! - Cornerness threshold.
    //! - Threshold used to suppress landmarks on edges.
    CVolumeCornerDetector(tSize Size, double dThreshold, double dEdgeThreshold)
        : m_WindowSize(Size)
        , m_dThreshold(dThreshold)
        , m_dEdgeThreshold(dEdgeThreshold)
    {
        MDS_ASSERT(Size > 0 && mds::math::isOdd(Size)
            && dThreshold >= 0.0 && dThreshold <= 1.0
            && dEdgeThreshold >= 0.0 && dEdgeThreshold <= 1.0
            );
    }

    //! Destructor.
    ~CVolumeCornerDetector() {}


    //! Volume landmark detection.
    //! - Returns false on failure.
    bool operator()(const V& SrcVolume, tCorners& Corners);

    //! Evaluates cornerness function of all voxels.
    //! - Returns false on failure. 
    bool operator()(const V& SrcVolume, V& DstVolume);

    //! Returns cornerness threshold.
    double getThreshold() { return m_dThreshold; }

    //! Sets cornerness threshold.
    inline void setThreshold(double dThreshold);

    //! Returns threshold used for suppression of landmarks lying on edges.
    double getEdgeThreshold() { return m_dEdgeThreshold; }

    //! Sets threshold used for suppression of landmarks lying on edges.
    inline void setEdgeThreshold(double dEdgeThreshold);

protected:
    //! Sobel operators.
//    CVolumeFilter<CFVolume, VF_SOBEL_X> m_SobelX;
//    CVolumeFilter<CFVolume, VF_SOBEL_Y> m_SobelY;
//    CVolumeFilter<CFVolume, VF_SOBEL_Z> m_SobelZ;
    CVolumeFilter<CFVolume, VF_DIFF_X> m_SobelX;
    CVolumeFilter<CFVolume, VF_DIFF_Y> m_SobelY;
    CVolumeFilter<CFVolume, VF_DIFF_Z> m_SobelZ;

    //! Size of the window used to estimate eigen values.
    tSize m_WindowSize;

    //! Corner detector parameters.
    double m_dThreshold, m_dEdgeThreshold;

protected:
    //! Returns true if any neighbouring voxel has greater value
    //! than a given threshold.
    inline bool checkNeighbours(CFVolume& Volume,
                                tSize x,
                                tSize y,
                                tSize z,
                                tFloatPixel T
                                );
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsRohr.hxx"


} // namespace img
} // namespace mds

#endif // MDS_VCD_ROHR_H

