//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFunctions/mdsGeneral.hxx       \n
 * Section: libImage                            \n
 * Date:    2006/02/20                          \n
 *
 * $Id: mdsGeneral.hxx 64 2006-08-11 08:45:24Z spanel $
 *
 * Description:
 * - General image functions.
 */


//==============================================================================
/*
 * Functions templates.
 */

template <typename R, class I1, class I2>
inline R getProduct(const CImageBase<I1>& Image1, const CImageBase<I2>& Image2)
{
    typedef typename I1::tPixel tPixel1;
    typedef typename I2::tPixel tPixel2;
    const I1& Image1Impl = Image1.getImpl();
    const I2& Image2Impl = Image2.getImpl();

    tSize XCount = mds::math::getMin(Image1Impl.getXSize(), Image2Impl.getXSize());
    tSize YCount = mds::math::getMin(Image1Impl.getYSize(), Image2Impl.getYSize());

    tSize Count = XCount * YCount;
    MDS_CHECK(Count > 0, return R(0));

    R Total = R(0);
    for( tSize j = 0; j < YCount; ++j )
    {
        const tPixel1 *p1 = Image1Impl.getRowPtr(j);
        const tPixel1 *p1Max = p1 + XCount;
        const tPixel2 *p2 = Image2Impl.getRowPtr(j);
        while( p1 < p1Max )
        {
            Total += R(*(p2++)) * R(*(p1++));
        }
    }
    return Total;
}


template <typename R, class I1, class I2>
inline R getMeanSquareError(const CImageBase<I1>& Image1, const CImageBase<I2>& Image2)
{
    typedef typename I1::tPixel tPixel1;
    typedef typename I2::tPixel tPixel2;
    const I1& Image1Impl = Image1.getImpl();
    const I2& Image2Impl = Image2.getImpl();

    tSize XCount = mds::math::getMin(Image1Impl.getXSize(), Image2Impl.getXSize());
    tSize YCount = mds::math::getMin(Image1Impl.getYSize(), Image2Impl.getYSize());

    tSize Count = XCount * YCount;
    MDS_CHECK(Count > 0, return R(0));

    R Total = R(0);
    for( tSize j = 0; j < YCount; ++j )
    {
        const tPixel1 *p1 = Image1Impl.getRowPtr(j);
        const tPixel1 *p1Max = p1 + XCount;
        const tPixel2 *p2 = Image2Impl.getRowPtr(j);
        while( p1 < p1Max )
        {
            R Temp = R(*(p2++)) - R(*(p2++));
            Total += Temp * Temp;
        }
    }
    return (Total / R(Count));
}


template <typename R, class I>
inline R getSum(const CImageBase<I>& Image)
{
    typedef typename I::tPixel tPixel;
    const I& ImageImpl = Image.getImpl();

    R Total = R(0);
    for( tSize j = 0; j < ImageImpl.getYSize(); ++j )
    {
        const tPixel *p = ImageImpl.getRowPtr(j);
        const tPixel *pMax = p + ImageImpl.getXSize();
        while( p < pMax )
        {
            Total += R(*(p++));
        }
    }
    return Total;
}


template <typename R, class I>
inline R getSumOfSquares(const CImageBase<I>& Image)
{
    typedef typename I::tPixel tPixel;
    const I& ImageImpl = Image.getImpl();

    R Total = R(0);
    for( tSize j = 0; j < ImageImpl.getYSize(); ++j )
    {
        const tPixel *p = ImageImpl.getRowPtr(j);
        const tPixel *pMax = p + ImageImpl.getXSize();
        while( p < pMax )
        {
            R Sample = R(*(p++));
            Total += Sample * Sample;
        }
    }
    return Total;
}


template <typename R, class I>
inline R getMean(const CImageBase<I>& Image)
{
    typedef typename I::tPixel tPixel;
    const I& ImageImpl = Image.getImpl();

    tSize Count = ImageImpl.getYSize() * ImageImpl.getXSize();
    MDS_CHECK(Count > 0, return R(0));

    R Total = R(0);
    for( tSize j = 0; j < ImageImpl.getYSize(); j++ )
    {
        const tPixel *p = ImageImpl.getRowPtr(j);
        const tPixel *pMax = p + ImageImpl.getXSize();
        while( p < pMax )
        {
            Total += R(*(p++));
        }
    }
    return Total / R(Count);
}


template <typename R, class I>
inline R getVariance(const CImageBase<I>& Image)
{
    typedef typename I::tPixel tPixel;
    const I& ImageImpl = Image.getImpl();

    tSize Count = ImageImpl.getYSize() * ImageImpl.getXSize();
    MDS_CHECK(Count > 0, return R(0));

    R Sum = R(0), SumSqr = R(0);
    for( tSize j = 0; j < ImageImpl.getYSize(); ++j )
    {
        const tPixel *p = ImageImpl.getRowPtr(j);
        const tPixel *pMax = p + ImageImpl.getXSize();
        while( p < pMax )
        {
            R Temp = R(*(p++));
            Sum += Temp;
            SumSqr += Temp * Temp;
        }
    }
    R Temp = R(1) / R(Count);
    return SumSqr * Temp - (Sum * Temp * Sum * Temp);
}


template <typename R, class I>
inline R getMin(const CImageBase<I>& Image)
{
    typedef typename I::tPixel tPixel;
    const I& ImageImpl = Image.getImpl();

    tSize Count = ImageImpl.getYSize() * ImageImpl.getXSize();
    MDS_CHECK(Count > 0, return R(0));

    tPixel MinVal = *ImageImpl.getPtr();
    for( tSize j = 0; j < ImageImpl.getYSize(); ++j )
    {
        const tPixel *p = ImageImpl.getRowPtr(j);
        const tPixel *pMax = p + ImageImpl.getXSize();
        while( p < pMax )
        {
            if( *p < MinVal )
            {
                MinVal = *p;
            }
            ++p;
        }
    }
    return R(MinVal);
}


template <typename R, class I>
inline R getMax(const CImageBase<I>& Image)
{
    typedef typename I::tPixel tPixel;
    const I& ImageImpl = Image.getImpl();

    tSize Count = ImageImpl.getYSize() * ImageImpl.getXSize();
    MDS_CHECK(Count > 0, return R(0));

    tPixel MaxVal = *ImageImpl.getPtr();
    for( tSize j = 0; j < ImageImpl.getYSize(); ++j )
    {
        const tPixel *p = ImageImpl.getRowPtr(j);
        const tPixel *pMax = p + ImageImpl.getXSize();
        while( p < pMax )
        {
            if( *p > MaxVal )
            {
                MaxVal = *p;
            }
            ++p;
        }
    }
    return R(MaxVal);
}

