//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFunctions/mdsConvolution.hxx   \n
 * Section: libImage                            \n
 * Date:    2006/02/20                          \n
 *
 * $Id: mdsConvolution.hxx 64 2006-08-11 08:45:24Z spanel $
 *
 * Description:
 * - Image convolution functions.
 */


//==============================================================================
/*
 * Functions templates.
 */

template <typename R, class I>
inline R convolve(const CImageBase<I>& Image,
                  tSize x,
                  tSize y,
                  const CKernel2D& Kernel
                 )
{
    typedef typename I::tPixel tPixel;
    const I& ImageImpl = Image.getImpl();

    tSize Size = Kernel.getSize();
    tSize Half = Size >> 1;
    tSize Offset = ImageImpl.getYOffset() - Size;

    const tPixel *p = ImageImpl.getPtr(x - Half, y - Half);
    const CKernel2D::tData *pK = Kernel.getPtr();

    R Sum = R(0);
    for( tSize j = 0; j < Size; ++j )
    {
        const tPixel *pMax = p + Size;
        while( p < pMax )
        {
            Sum += R(*(p++)) * R(*(pK++));
        }
        p += Offset;
    }
    return Sum;
}


template <typename R, class I>
inline R convolveX(const CImageBase<I>& Image,
                   tSize x,
                   tSize y,
                   const CKernel1D& Kernel
                  )
{
    typedef typename I::tPixel tPixel;
    const I& ImageImpl = Image.getImpl();

    tSize Size = Kernel.getSize();
    tSize Half = Size >> 1;

    const tPixel *p = ImageImpl.getPtr(x - Half, y);
    const tPixel *pMax = p + Size;
    const CKernel1D::tData *pK = Kernel.getPtr();

    R Sum = R(0);
    while( p < pMax)
    {
        Sum += R(*(p++)) * R(*(pK++));
    }
    return Sum;
}


template <typename R, class I>
inline R convolveY(const CImageBase<I>& Image,
                   tSize x,
                   tSize y,
                   const CKernel1D& Kernel
                  )
{
    typedef typename I::tPixel tPixel;
    const I& ImageImpl = Image.getImpl();

    tSize Size = Kernel.getSize();
    tSize Half = Size >> 1;

    const tPixel *p = ImageImpl.getPtr(x, y - Half);
    const tPixel *pMax = p + Size * ImageImpl.getYOffset();
    const CKernel1D::tData *pK = Kernel.getPtr();

    R Sum = R(0);
    while( p < pMax)
    {
        Sum += R(*(p)) * R(*(pK++));
        p += ImageImpl.getYOffset();
    }
    return Sum;
}


template <typename R, class I>
inline R convolve(const CImageBase<I>& Image,
                  const CPoint3D& Point,
                  const CKernel2D& Kernel
                 )
{
    typedef typename I::tPixel tPixel;
    const I& ImageImpl = Image.getImpl();

    tSize Size = Kernel.getSize();
    tSize Half = Size >> 1;

    CPoint3D Temp(0, Point.y - Half);
    R Sum = R(0);
    for( tSize m = 0; m < Size; ++m )
    {
        Temp.x = Point.x - Half;
        for( tSize l = 0; l < Size; ++l )
        {
            Sum += R(ImageImpl.interpolate(Temp)) * R(Kernel.get(l, m));
            Temp.x += 1.0f;
        }
        Temp.y += 1.0f;
    }
    return Sum;
}


template <typename R, class I>
inline R convolveX(const CImageBase<I>& Image,
                   const CPoint3D& Point,
                   const CKernel1D& Kernel
                  )
{
    typedef typename I::tPixel tPixel;
    const I& ImageImpl = Image.getImpl();

    tSize Size = Kernel.getSize();
    tSize Half = Size >> 1;

    CPoint3D Temp(Point.x - Half, Point.y);
    R Sum = R(0);
    for( tSize l = 0; l < Size; ++l )
    {
        Sum += R(ImageImpl.interpolate(Temp)) * R(Kernel.get(l));
        Temp.x += 1.0;
    }
    return Sum;
}


template <typename R, class I>
inline R convolveY(const CImageBase<I>& Image,
                   const CPoint3D& Point,
                   const CKernel1D& Kernel
                  )
{
    typedef typename I::tPixel tPixel;
    const I& ImageImpl = Image.getImpl();

    tSize Size = Kernel.getSize();
    tSize Half = Size >> 1;

    CPoint3D Temp(Point.x, Point.y - Half);
    R Sum = R(0);
    for( tSize m = 0; m < Size; ++m )
    {
        Sum += R(ImageImpl.interpolate(Temp)) * R(Kernel.get(m));
        Temp.y += 1.0;
    }
    return Sum;
}

