//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFunctions/mdsConversion.hxx    \n
 * Section: libImage                            \n
 * Date:    2006/08/09                          \n
 *
 * $Id: mdsConversion.hxx 119 2006-10-30 18:23:00Z spanel $
 *
 * Description:
 * - Image conversion functions.
 */


//==============================================================================
/*
 * Functions templates.
 */

template <class I1, class I2>
inline bool convert(const CImageBase<I1>& SrcImage, CImageBase<I2>& DstImage)
{
    typedef typename I1::tPixel tPixel1;
    typedef typename I2::tPixel tPixel2;
    const I1& SrcImageImpl = SrcImage.getImpl();
    I2& DstImageImpl = DstImage.getImpl();

    // Get size and margin of the source image
    tSize XSize = SrcImageImpl.getXSize();
    tSize YSize = SrcImageImpl.getYSize();
    tSize Margin = SrcImageImpl.getMargin();

    // Resize the destination image
    DstImageImpl.create(XSize, YSize, Margin);

    // Convert the source image
    for( tSize j = 0; j < YSize; ++j )
    {
        const tPixel1 *p1 = SrcImageImpl.getRowPtr(j);
        const tPixel1 *p1Max = p1 + XSize;
        tPixel2 *p2 = DstImageImpl.getRowPtr(j);
        while( p1 < p1Max )
        {
            *p2 = CPixelConversion<tPixel1,tPixel2>::convert(*p1);
            p1 += SrcImageImpl.getXOffset();
            p2 += DstImageImpl.getXOffset();
        }
    }
    
    // O.K.
    return true;
}


template <class I>
inline void log(CImageBase<I>& Image)
{
    typedef typename I::tPixel tPixel;
    I& ImageImpl = Image.getImpl();

    static const tPixel PMin = CPixelTraits<tPixel>::getPixelMin();
    static const tPixel PRange = CPixelTraits<tPixel>::getPixelMax() - PMin;

    // Get size of the image
    tSize XSize = ImageImpl.getXSize();
    tSize YSize = ImageImpl.getYSize();

    // Maximal and minimal value in the image
    tPixel Max = getMax<tPixel>(Image);
    tPixel Min = getMin<tPixel>(Image);

    // Estimate the scaling constant
    double dScale = double(PRange) / std::log(1.0 + Max - Min);

    // Logarithmic operator
    for( tSize j = 0; j < YSize; ++j )
    {
        tPixel *p = ImageImpl.getRowPtr(j);
        tPixel *pMax = p + XSize;
        for( ; p < pMax; ++p )
        {
            *p = tPixel(std::log(1.0 + *p - Min) * dScale + PMin);
        }
    }
}


template <class I>
inline bool real(const CImageBase<CComplexImage>& SrcImage, CImageBase<I>& DstImage)
{
    typedef typename CComplexImage::tPixel tPixel1;
    typedef typename CComplexImage::tPixel::tComponent tComponent;
    typedef typename I::tPixel tPixel2;
    
    const CComplexImage& SrcImageImpl = SrcImage.getImpl();
    I& DstImageImpl = DstImage.getImpl();

    // Get size and margin of the source image
    tSize XSize = SrcImageImpl.getXSize();
    tSize YSize = SrcImageImpl.getYSize();
    tSize Margin = SrcImageImpl.getMargin();

    // Resize the destination image
    DstImageImpl.create(XSize, YSize, Margin);
    
    // Convert the source image
    for( tSize j = 0; j < YSize; ++j )
    {
        const tPixel1 *p1 = SrcImageImpl.getRowPtr(j);
        const tPixel1 *p1Max = p1 + XSize;
        tPixel2 *p2 = DstImageImpl.getRowPtr(j);
        while( p1 < p1Max )
        {
            *p2 = CPixelConversion<tComponent,tPixel2>::convert(p1->getReal());
            p1 += SrcImageImpl.getXOffset();
            p2 += DstImageImpl.getXOffset();
        }
    }
    
    // O.K.
    return true;
}


template <class I>
inline bool imag(const CImageBase<CComplexImage>& SrcImage, CImageBase<I>& DstImage)
{
    typedef typename CComplexImage::tPixel tPixel1;
    typedef typename CComplexImage::tPixel::tComponent tComponent;
    typedef typename I::tPixel tPixel2;
    
    const CComplexImage& SrcImageImpl = SrcImage.getImpl();
    I& DstImageImpl = DstImage.getImpl();

    // Get size and margin of the source image
    tSize XSize = SrcImageImpl.getXSize();
    tSize YSize = SrcImageImpl.getYSize();
    tSize Margin = SrcImageImpl.getMargin();

    // Resize the destination image
    DstImageImpl.create(XSize, YSize, Margin);
    
    // Convert the source image
    for( tSize j = 0; j < YSize; ++j )
    {
        const tPixel1 *p1 = SrcImageImpl.getRowPtr(j);
        const tPixel1 *p1Max = p1 + XSize;
        tPixel2 *p2 = DstImageImpl.getRowPtr(j);
        while( p1 < p1Max )
        {
            *p2 = CPixelConversion<tComponent,tPixel2>::convert(p1->getImag());
            p1 += SrcImageImpl.getXOffset();
            p2 += DstImageImpl.getXOffset();
        }
    }
    
    // O.K.
    return true;
}


template <class I>
inline bool abs(const CImageBase<CComplexImage>& SrcImage, CImageBase<I>& DstImage)
{
    typedef typename CComplexImage::tPixel tPixel1;
    typedef typename CComplexImage::tPixel::tComponent tComponent;
    typedef typename I::tPixel tPixel2;
    
    const CComplexImage& SrcImageImpl = SrcImage.getImpl();
    I& DstImageImpl = DstImage.getImpl();

    // Get size and margin of the source image
    tSize XSize = SrcImageImpl.getXSize();
    tSize YSize = SrcImageImpl.getYSize();
    tSize Margin = SrcImageImpl.getMargin();

    // Resize the destination image
    DstImageImpl.create(XSize, YSize, Margin);
    
    // Convert the source image
    for( tSize j = 0; j < YSize; ++j )
    {
        const tPixel1 *p1 = SrcImageImpl.getRowPtr(j);
        const tPixel1 *p1Max = p1 + XSize;
        tPixel2 *p2 = DstImageImpl.getRowPtr(j);
        while( p1 < p1Max )
        {
            *p2 = CPixelConversion<tComponent,tPixel2>::convert(p1->getAbs());
            p1 += SrcImageImpl.getXOffset();
            p2 += DstImageImpl.getXOffset();
        }
    }
    
    // O.K.
    return true;
}


template <class I>
inline bool arg(const CImageBase<CComplexImage>& SrcImage, CImageBase<I>& DstImage)
{
    typedef typename CComplexImage::tPixel tPixel1;
    typedef typename CComplexImage::tPixel::tComponent tComponent;
    typedef typename I::tPixel tPixel2;
    
    const CComplexImage& SrcImageImpl = SrcImage.getImpl();
    I& DstImageImpl = DstImage.getImpl();

    // Get size and margin of the source image
    tSize XSize = SrcImageImpl.getXSize();
    tSize YSize = SrcImageImpl.getYSize();
    tSize Margin = SrcImageImpl.getMargin();

    // Resize the destination image
    DstImageImpl.create(XSize, YSize, Margin);
    
    // Convert the source image
    for( tSize j = 0; j < YSize; ++j )
    {
        const tPixel1 *p1 = SrcImageImpl.getRowPtr(j);
        const tPixel1 *p1Max = p1 + XSize;
        tPixel2 *p2 = DstImageImpl.getRowPtr(j);
        while( p1 < p1Max )
        {
            *p2 = CPixelConversion<tComponent,tPixel2>::convert(p1->getArg());
            p1 += SrcImageImpl.getXOffset();
            p2 += DstImageImpl.getXOffset();
        }
    }

    // O.K.
    return true;
}

