//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFilters/mdsSobel.h             \n
 * Section: libImage                            \n
 * Date:    2005/09/15                          \n
 *
 * $Id: mdsSobel.h 185 2006-12-15 11:48:24Z spanel $
 *
 * Description:
 * - Sobel operator.
 */

#ifndef MDS_IF_SOBEL_H
#define MDS_IF_SOBEL_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Math/mdsBase.h>

#include "../mdsImageFilter.h"


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Sobel operator
MDS_DECLARE_ID(IF_SOBEL_X);
MDS_DECLARE_ID(IF_SOBEL_Y);


//==============================================================================
/*!
 * Sobel operator (horizontal version).
 * - Parameter I is a used image type.
 * - Kernel definition: \n
 *   1 [ -1  0  1 ] \n
 *   - [ -2  0  2 ] \n
 *   4 [ -1  0  1 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_SOBEL_X, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response
    static const int DENOM = 4;

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    inline tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 3; }
};


//==============================================================================
/*!
 * Sobel operator (vertical edges).
 * - Parameter I is a used image type.
 * - Kernel definition: \n
 *   1 [ -1 -2 -1 ] \n
 *   - [  0  0  0 ] \n
 *   4 [  1  2  1 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_SOBEL_Y, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response
    static const int DENOM = 4;

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    inline tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 3; }
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsSobel.hxx"


} // namespace img
} // namespace mds

#endif // MDS_IF_SOBEL_H

