//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFilters/mdsRoberts.h           \n
 * Section: libImage                            \n
 * Date:    2005/09/15                          \n
 *
 * $Id: mdsRoberts.h 90 2006-09-13 12:22:57Z spanel $
 * 
 * Description:
 * - Roberts cross operator.
 */

#ifndef MDS_IF_ROBERTS_H
#define MDS_IF_ROBERTS_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Math/mdsBase.h>

#include "../mdsImageFilter.h"


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Roberts cross operators.
MDS_DECLARE_ID(IF_ROBERTS_X);
MDS_DECLARE_ID(IF_ROBERTS_Y);


//==============================================================================
/*!
 * Roberts cross operator (first version).
 * - Parameter I is a used image type.
 * - Policy N is used for filter response normalization.
 * - Kernel definition: \n
 *   [ 1  0 ] \n
 *   [ 0 -1 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_ROBERTS_X, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response
    static const int DENOM = 1;

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 2; }
};


//==============================================================================
/*!
 * Roberts operator (second version).
 * - Parameter I is a used image type.
 * - Policy N is used for filter response normalization.
 * - Kernel definition: \n
 *   [  0  1 ] \n
 *   [ -1  0 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_ROBERTS_Y, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response
    static const int DENOM = 1;

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 2; }
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsRoberts.hxx"


} // namespace img
} // namespace mds

#endif // MDS_IF_ROBERTS_H

