//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFilters/mdsMinMax.hxx          \n
 * Section: libImage                            \n
 * Date:    2006/06/14                          \n
 *
 * $Id: mdsMinMax.hxx 90 2006-09-13 12:22:57Z spanel $
 * 
 * Description:
 * - Square min and max image filters.
 */


//==============================================================================
/*
 * Global functions.
 */

namespace minmax
{

//! Finds minimal value.
template <typename T>
T findMin(T *pData, tSize Size)
{
    MDS_CHECK(Size > 0, return T(0));

    T Min = pData[0];

    T *p = pData + 1;
    T *pMax = pData + Size;
    while( p < pMax )
    {
        if( *p < Min )
        {
            Min = *p;
        }
        ++p;
    }

    return Min;
}

//! Finds maximal value.
template <typename T>
T findMax(T *pData, tSize Size)
{
    MDS_CHECK(Size > 0, return T(0));

    T Max = pData[0];

    T *p = pData + 1;
    T *pMax = pData + Size;
    while( p < pMax )
    {
        if( *p > Max )
        {
            Max = *p;
        }
        ++p;
    }

    return Max;
}

} // namespace


//==============================================================================
/*
 * Methods templates.
 */

// Image filtering method
template <class I, template <typename> class N>
bool CImageFilter<I, IF_MIN, N>::operator()(const I& SrcImage, I& DstImage)
{
    // Image size
    tSize XCount = mds::math::getMin(SrcImage.getXSize(), DstImage.getXSize());
    tSize YCount = mds::math::getMin(SrcImage.getYSize(), DstImage.getYSize());

    // Filter the image
    for( tSize y = 0; y < YCount; ++y )
    {
        for( tSize x = 0; x < XCount; ++x )
        {
            // Copy pixels from the window
            SrcImage.copyWindow(x, y, m_MinSize, m_Data.getPtr());

            // Min finding
            tPixel Min = minmax::findMin<tPixel>(m_Data.getPtr(), m_Data.getSize());

            // Set pixel value
            DstImage.set(x, y, Min);
        }
    }

    // O.K.
    return true;
}


// Image filter response
template <class I, template <typename> class N>
typename CImageFilter<I, IF_MIN, N>::tResult CImageFilter<I, IF_MIN, N>::getResponse(const I& SrcImage, tSize x, tSize y)
{
    // Copy pixels from the window
    SrcImage.copyWindow(x, y, m_MinSize, m_Data.getPtr());

    // Compute filter response
    return tResult(minmax::findMin<tPixel>(m_Data.getPtr(), m_Data.getSize()));
}


// Image filtering method
template <class I, template <typename> class N>
bool CImageFilter<I, IF_MAX, N>::operator()(const I& SrcImage, I& DstImage)
{
    // Image size
    tSize XCount = mds::math::getMin(SrcImage.getXSize(), DstImage.getXSize());
    tSize YCount = mds::math::getMin(SrcImage.getYSize(), DstImage.getYSize());

    // Filter the image
    for( tSize y = 0; y < YCount; ++y )
    {
        for( tSize x = 0; x < XCount; ++x )
        {
            // Copy pixels from the window
            SrcImage.copyWindow(x, y, m_MaxSize, m_Data.getPtr());

            // Min finding
            tPixel Max = minmax::findMax<tPixel>(m_Data.getPtr(), m_Data.getSize());

            // Set pixel value
            DstImage.set(x, y, Max);
        }
    }

    // O.K.
    return true;
}


// Image filter response
template <class I, template <typename> class N>
typename CImageFilter<I, IF_MAX, N>::tResult CImageFilter<I, IF_MAX, N>::getResponse(const I& SrcImage, tSize x, tSize y)
{
    // Copy pixels from the window
    SrcImage.copyWindow(x, y, m_MaxSize, m_Data.getPtr());

    // Compute filter response
    return tResult(minmax::findMax<tPixel>(m_Data.getPtr(), m_Data.getSize()));
}

