//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFilters/mdsMinMax.h            \n
 * Section: libImage                            \n
 * Date:    2006/06/14                          \n
 *
 * $Id: mdsMinMax.h 90 2006-09-13 12:22:57Z spanel $
 * 
 * Description:
 * - Square min and max image filters.
 */

#ifndef MDS_IF_MINMAX_H
#define MDS_IF_MINMAX_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Math/mdsBase.h>
#include <MDSTk/Math/mdsVector.h>

#include "../mdsImageFilter.h"


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Square min image filter.
MDS_DECLARE_ID(IF_MIN);

//! Square max image filter.
MDS_DECLARE_ID(IF_MAX);


//==============================================================================
/*!
 * Template providing square min-value image filtering functions.
 * - Parameter I is a used image type.
 * - Policy N is used for filter response normalization.
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_MIN, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

public:
    //! Constructor that creates a new filter.
    //! - Parameter 'Size' is a window size and it must be an odd number.
    CImageFilter(tSize Size)
            : m_Data(Size * Size)
            , m_MinSize(Size)
    {
        MDS_ASSERT((Size % 2) == 1);
    }

    //! Destructor
    ~CImageFilter() {}


    //! Min image filtering.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns current filter size.
    tSize getSize() const { return m_MinSize; }

    //! Sets the windows size
    void resize(tSize Size)
    {
        MDS_ASSERT((Size % 2) == 1);

        m_Data.resize(Size * Size);
        m_MinSize = Size;
    }

protected:
    //! Data vector used to find min value.
    mds::math::CVector<tPixel> m_Data;

    //! Min filter size.
    tSize m_MinSize;
};


//==============================================================================
/*!
 * Template providing square max-value image filtering functions.
 * - Parameter I is a used image type.
 * - Policy N is used for filter response normalization.
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_MAX, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

public:
    //! Constructor that creates a new filter.
    //! - Parameter 'Size' is a window size and it must be an odd number.
    CImageFilter(tSize Size)
            : m_Data(Size * Size)
            , m_MaxSize(Size)
    {
        MDS_ASSERT((Size % 2) == 1);
    }

    //! Destructor
    ~CImageFilter() {}


    //! Max image filtering.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns current filter size.
    tSize getSize() const { return m_MaxSize; }

    //! Sets the windows size.
    void resize(tSize Size)
    {
        MDS_ASSERT((Size % 2) == 1);

        m_Data.resize(Size * Size);
        m_MaxSize = Size;
    }

protected:
    //! Data vector used to find max value.
    mds::math::CVector<tPixel> m_Data;

    //! Max filter size.
    tSize m_MaxSize;
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsMinMax.hxx"


} // namespace img
} // namespace mds

#endif // MDS_IF_MINMAX_H

