//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFilters/mdsMedian.hxx          \n
 * Section: libImage                            \n
 * Date:    2005/09/06                          \n
 *
 * $Id: mdsMedian.hxx 90 2006-09-13 12:22:57Z spanel $
 * 
 * Description:
 * - Median image filter.
 */


//==============================================================================
/*
 * Global functions.
 */

namespace median
{

//! Median value finding (Z Algorithm)
template <typename T>
T findMedian(T *pData, tSize Size)
{
    // Index of the median value
    const tSize Median = Size / 2;

    // Z algorithm initialization
    tSize Left = 0;
    tSize Right = Size - 1;

    // Median finding
    while( Left < Right )
    {
        T Value = pData[Median];
        tSize LeftValue = Left;
        tSize RightValue = Right;

        while( RightValue >= Median && Median >= LeftValue )
        {
            while( pData[LeftValue] < Value )
            {
                ++LeftValue;
            }
            while( Value < pData[RightValue] )
            {
                --RightValue;
            }

            T Temp = pData[LeftValue];
            pData[LeftValue] = pData[RightValue];
            pData[RightValue] = Temp;

            ++LeftValue;
            --RightValue;
        }

        if( RightValue < Median )
        {
            Left = LeftValue;
        }
        if( Median < LeftValue )
        {
            Right = RightValue;
        }
    }

    return pData[Median];
}

} // namespace


//==============================================================================
/*
 * Methods templates.
 */

// Image filtering method
template <typename I, template <typename> class N>
bool CImageFilter<I, IF_MEDIAN, N>::operator()(const I& SrcImage, I& DstImage)
{
    // Image size
    tSize XCount = mds::math::getMin(SrcImage.getXSize(), DstImage.getXSize());
    tSize YCount = mds::math::getMin(SrcImage.getYSize(), DstImage.getYSize());

    // Filter the image
    for( tSize y = 0; y < YCount; ++y )
    {
        for( tSize x = 0; x < XCount; ++x )
        {
            // Copy pixels from the window
            SrcImage.copyWindow(x, y, m_MedianSize, m_Data.getPtr());

            // Median finding
            tPixel Median = median::findMedian<tPixel>(m_Data.getPtr(), m_Data.getSize());

            // Set pixel value
            DstImage.set(x, y, Median);
        }
    }

    // O.K.
    return true;
}


// Image filter response
template <typename I, template <typename> class N>
typename CImageFilter<I, IF_MEDIAN, N>::tResult CImageFilter<I, IF_MEDIAN, N>::getResponse(const I& SrcImage, tSize x, tSize y)
{
    // Copy pixels from the window
    SrcImage.copyWindow(x, y, m_MedianSize, m_Data.getPtr());

    // Compute filter response
    return tResult(median::findMedian<tPixel>(m_Data.getPtr(), m_Data.getSize()));
}

