//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFilters/mdsMedian.h            \n
 * Section: libImage                            \n
 * Date:    2005/09/06                          \n
 *
 * $Id: mdsMedian.h 90 2006-09-13 12:22:57Z spanel $
 * 
 * Description:
 * - Median image filter.
 */

#ifndef MDS_IF_MEDIAN_H
#define MDS_IF_MEDIAN_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsTypes.h>
#include <MDSTk/Math/mdsBase.h>
#include <MDSTk/Math/mdsVector.h>

#include "../mdsImageFilter.h"

//#include <cstdlib>


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Median image filter
MDS_DECLARE_ID(IF_MEDIAN);


//==============================================================================
/*!
 * Template providing median image filtering functions.
 * - Parameter I is a used image type.
 * - Policy N is used for filter response normalization.
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_MEDIAN, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

public:
    //! Constructor that creates a new median filter.
    //! - Parameter 'Size' is a window size and it must be an odd number.
    CImageFilter(tSize Size)
        : m_Data(Size * Size)
        , m_MedianSize(Size)
    {
        MDS_ASSERT((Size % 2) == 1);
    }

    //! Destructor
    ~CImageFilter() {}


    //! Median image filtering
    //! - Returns false on failure
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns current median size
    tSize getSize() const { return m_MedianSize; }

    //! Sets the windows size
    void resize(tSize Size)
    {
        MDS_ASSERT((Size % 2) == 1);

        m_Data.resize(Size * Size);
        m_MedianSize = Size;
    }

protected:
    //! Data vector used to find median value.
    mds::math::CVector<tPixel> m_Data;

    //! Median filter size
    tSize m_MedianSize;
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsMedian.hxx"


} // namespace img
} // namespace mds

#endif // MDS_IF_MEDIAN_H

