//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2006 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFilters/mdsLocalMoments.h      \n
 * Section: libImage                            \n
 * Date:    2006/03/30                          \n
 *
 * $Id: mdsLocalMoments.h 90 2006-09-13 12:22:57Z spanel $
 *
 * Description:
 * - Local moments (local texture features).
 */

#ifndef MDS_IF_LOCAL_MOMENTS_H
#define MDS_IF_LOCAL_MOMENTS_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Math/mdsBase.h>

#include "../mdsImageFilter.h"


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Local moments.
MDS_DECLARE_ID(IF_LOCAL_MOMENT_00);
MDS_DECLARE_ID(IF_LOCAL_MOMENT_10);
MDS_DECLARE_ID(IF_LOCAL_MOMENT_01);
MDS_DECLARE_ID(IF_LOCAL_MOMENT_11);
MDS_DECLARE_ID(IF_LOCAL_MOMENT_20);
MDS_DECLARE_ID(IF_LOCAL_MOMENT_02);
MDS_DECLARE_ID(IF_LOCAL_MOMENT_22);


//==============================================================================
/*!
 * Local moment.
 * - Parameter I is a used image type.
 * - Kernel definition: \n
 *   [ 1  1  1 ] \n
 *   [ 1  1  1 ] \n
 *   [ 1  1  1 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_LOCAL_MOMENT_00, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response.
    static const int DENOM = 9;

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    inline tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 3; }
};


//==============================================================================
/*!
 * Local moment.
 * - Parameter I is a used image type.
 * - Kernel definition: \n
 *   [ -1 -1 -1 ] \n
 *   [  0  0  0 ] \n
 *   [  1  1  1 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_LOCAL_MOMENT_10, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response.
    static const int DENOM = 3;

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    inline tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 3; }
};


//==============================================================================
/*!
 * Local moment.
 * - Parameter I is a used image type.
 * - Kernel definition: \n
 *   [ -1  0  1 ] \n
 *   [ -1  0  1 ] \n
 *   [ -1  0  1 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_LOCAL_MOMENT_01, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response.
    static const int DENOM = 3;

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    inline tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 3; }
};


//==============================================================================
/*!
 * Local moment.
 * - Parameter I is a used image type.
 * - Kernel definition: \n
 *   [  1  0 -1 ] \n
 *   [  0  0  0 ] \n
 *   [ -1  0  1 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_LOCAL_MOMENT_11, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response.
    static const int DENOM = 2;

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    inline tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 3; }
};


//==============================================================================
/*!
 * Local moment.
 * - Parameter I is a used image type.
 * - Kernel definition: \n
 *   [ 1  1  1 ] \n
 *   [ 0  0  0 ] \n
 *   [ 1  1  1 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_LOCAL_MOMENT_20, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response.
    static const int DENOM = 6;

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    inline tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 3; }
};


//==============================================================================
/*!
 * Local moment.
 * - Parameter I is a used image type.
 * - Kernel definition: \n
 *   [ 1  0  1 ] \n
 *   [ 1  0  1 ] \n
 *   [ 1  0  1 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_LOCAL_MOMENT_02, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response.
    static const int DENOM = 6;

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    inline tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 3; }
};


//==============================================================================
/*!
 * Local moment.
 * - Parameter I is a used image type.
 * - Kernel definition: \n
 *   [ 1  0  1 ] \n
 *   [ 0  0  0 ] \n
 *   [ 1  0  1 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_LOCAL_MOMENT_22, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response.
    static const int DENOM = 4;

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    inline tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 3; }
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsLocalMoments.hxx"


} // namespace img
} // namespace mds

#endif // MDS_IF_LOCAL_MOMENTS_H

