//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFilters/mdsLaplacian.h         \n
 * Section: libImage                            \n
 * Date:    2005/09/08                          \n
 *
 * $Id: mdsLaplacian.hxx 90 2006-09-13 12:22:57Z spanel $
 * 
 * Description:
 * - Laplacian image filters.
 */


//==============================================================================
/*
 * Methods templates.
 */

// Image filtering method
template <class I, template <typename> class N>
bool CImageFilter<I, IF_LAPLACIAN0, N>::operator()(const I& SrcImage, I& DstImage)
{
    // Image size
    tSize XCount = mds::math::getMin(SrcImage.getXSize(), DstImage.getXSize());
    tSize YCount = mds::math::getMin(SrcImage.getYSize(), DstImage.getYSize());

    // Filter the image
    for( tSize y = 0; y < YCount; ++y )
    {
        for( tSize x = 0; x < XCount; ++x )
        {
            tResult Value = getResponse(SrcImage, x, y);
            DstImage.set(x, y, norm::normalize(Value));
        }
    }

    // O.K.
    return true;
}


// Image filter response
template <class I, template <typename> class N>
typename CImageFilter<I, IF_LAPLACIAN0, N>::tResult CImageFilter<I, IF_LAPLACIAN0, N>::getResponse(const I& SrcImage, tSize x, tSize y)
{
    // Output normalization
    static const tResult Denom = 1.0f / DENOM;

    // Compute filter response
    tResult Value = Denom * (
                        tResult(SrcImage.get(x, y - 1))
                        + tResult(SrcImage.get(x, y + 1))
                        + tResult(SrcImage.get(x - 1, y))
                        + tResult(SrcImage.get(x + 1, y))
                        - 4 * tResult(SrcImage.get(x, y))
                        );
    return Value;
}


// Image filtering method
template <class I, template <typename> class N>
bool CImageFilter<I, IF_LAPLACIAN1, N>::operator()(const I& SrcImage, I& DstImage)
{
    // Image size
    tSize XCount = mds::math::getMin(SrcImage.getXSize(), DstImage.getXSize());
    tSize YCount = mds::math::getMin(SrcImage.getYSize(), DstImage.getYSize());

    // Filter the image
    for( tSize y = 0; y < YCount; ++y )
    {
        for( tSize x = 0; x < XCount; ++x )
        {
            tResult Value = getResponse(SrcImage, x, y);
            DstImage.set(x, y, norm::normalize(Value));
        }
    }

    // O.K.
    return true;
}


// Image filter response
template <class I, template <typename> class N>
typename CImageFilter<I, IF_LAPLACIAN1, N>::tResult CImageFilter<I, IF_LAPLACIAN1, N>::getResponse(const I& SrcImage, tSize x, tSize y)
{
    // Output normalization
    static const tResult Denom = 1.0f / DENOM;

    // Compute filter response
    tResult Value = Denom * (
                        tResult(SrcImage.get(x, y - 1))
                        + tResult(SrcImage.get(x, y + 1))
                        + tResult(SrcImage.get(x - 1, y))
                        + tResult(SrcImage.get(x - 1, y - 1))
                        + tResult(SrcImage.get(x - 1, y + 1))
                        + tResult(SrcImage.get(x + 1, y))
                        + tResult(SrcImage.get(x + 1, y - 1))
                        + tResult(SrcImage.get(x + 1, y + 1))
                        - 8 * tResult(SrcImage.get(x, y))
                        );
    return Value;
}


// Image filtering method
template <class I, template <typename> class N>
bool CImageFilter<I, IF_LAPLACIAN2, N>::operator()(const I& SrcImage, I& DstImage)
{
    // Image size
    tSize XCount = mds::math::getMin(SrcImage.getXSize(), DstImage.getXSize());
    tSize YCount = mds::math::getMin(SrcImage.getYSize(), DstImage.getYSize());

    // Filter the image
    for( tSize y = 0; y < YCount; ++y )
    {
        for( tSize x = 0; x < XCount; ++x )
        {
            tResult Value = getResponse(SrcImage, x, y);
            DstImage.set(x, y, norm::normalize(Value));
        }
    }

    // O.K.
    return true;
}


// Image filter response
template <class I, template <typename> class N>
typename CImageFilter<I, IF_LAPLACIAN2, N>::tResult CImageFilter<I, IF_LAPLACIAN2, N>::getResponse(const I& SrcImage, tSize x, tSize y)
{
    // Output normalization
    static const tResult Denom = 1.0f / DENOM;

    // Compute filter response
    tResult Value = Denom * (
                        2 * tResult(SrcImage.get(x, y - 1))
                        + 2 * tResult(SrcImage.get(x, y + 1))
                        + 2 * tResult(SrcImage.get(x - 1, y))
                        + 2 * tResult(SrcImage.get(x + 1, y))
                        - tResult(SrcImage.get(x - 1, y - 1))
                        - tResult(SrcImage.get(x - 1, y + 1))
                        - tResult(SrcImage.get(x + 1, y - 1))
                        - tResult(SrcImage.get(x + 1, y + 1))
                        - 4 * tResult(SrcImage.get(x, y))
                        );
    return Value;
}


// Filter kernel
template <class I, template <typename> class N>
const CKernel2D::tData CImageFilter<I, IF_LAPLACIAN_5, N>::KERNEL[] =
{
    0.0f,  0.0f, -1.0f,  0.0f,  0.0f,
    0.0f, -1.0f, -2.0f, -1.0f,  0.0f,
    -1.0f, -2.0f, 16.0f, -2.0f, -1.0f,
    0.0f, -1.0f, -2.0f, -1.0f,  0.0f,
    0.0f,  0.0f, -1.0f,  0.0f,  0.0f
};


// Image filtering method
template <class I, template <typename> class N>
bool CImageFilter<I, IF_LAPLACIAN_5, N>::operator()(const I& SrcImage, I& DstImage)
{
    // Image size
    tSize XCount = mds::math::getMin(SrcImage.getXSize(), DstImage.getXSize());
    tSize YCount = mds::math::getMin(SrcImage.getYSize(), DstImage.getYSize());

    // Filter the image
    for( tSize y = 0; y < YCount; ++y )
    {
        for( tSize x = 0; x < XCount; ++x )
        {
            tResult Value = getResponse(SrcImage, x, y);
            DstImage.set(x, y, norm::normalize(Value));
        }
    }

    // O.K.
    return true;
}


// Image filter response
template <class I, template <typename> class N>
typename CImageFilter<I, IF_LAPLACIAN_5, N>::tResult CImageFilter<I, IF_LAPLACIAN_5, N>::getResponse(const I& SrcImage, tSize x, tSize y)
{
    // Output normalization
    static const tResult Denom = 1.0 / DENOM;

    // Filter kernel
    static const CKernel2D Kernel(KERNEL, getSize());

    // Compute filter response
    return Denom * convolve<tResult>(SrcImage, x, y, Kernel);
}

