//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFilters/mdsLaplacian.h         \n
 * Section: libImage                            \n
 * Date:    2005/09/08                          \n
 *
 * $Id: mdsLaplacian.h 90 2006-09-13 12:22:57Z spanel $
 * 
 * Description:
 * - Laplacian image filters.
 */

#ifndef MDS_IF_LAPLACIAN_H
#define MDS_IF_LAPLACIAN_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Math/mdsBase.h>

#include "../mdsImageFilter.h"


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Laplacian image filters
MDS_DECLARE_ID(IF_LAPLACIAN0);
MDS_DECLARE_ID(IF_LAPLACIAN1);
MDS_DECLARE_ID(IF_LAPLACIAN2);
MDS_DECLARE_ID(IF_LAPLACIAN_5);


//==============================================================================
/*!
 * Laplacian image filter (first version).
 * - Parameter I is a used image type.
 * - Policy N is used for filter response normalization.
 * - Kernel definition: \n
 *   1 [ 0  1  0 ] \n
 *   - [ 1 -4  1 ] \n
 *   4 [ 0  1  0 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_LAPLACIAN0, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response
    static const int DENOM = 4;

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 3; }

};


//==============================================================================
/*!
 * Laplacian image filter (second version).
 * - Parameter I is a used image type.
 * - Policy N is used for filter response normalization.
 * - Kernel definition: \n
 *   1 [ 1  1  1 ] \n
 *   - [ 1 -8  1 ] \n
 *   8 [ 1  1  1 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_LAPLACIAN1, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response
    static const int DENOM = 8;

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 3; }
};


//==============================================================================
/*!
 * Laplacian image filter (third version).
 * - Parameter I is a used image type.
 * - Policy N is used for filter response normalization.
 * - Kernel definition: \n
 *   1 [ -1  2 -1 ] \n
 *   - [  2 -4  2 ] \n
 *   8 [ -1  2 -1 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_LAPLACIAN2, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

    //! Number used to divide filter response
    static const int DENOM = 8;

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Filter response normalization functor or function can be specified.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 3; }
};


//==============================================================================
/*!
 * Laplacian image filter (4-th version).
 * - Parameter I is a used image type.
 * - Policy N is used for filter response normalization.
 * - Kernel definition: \n
 *      [  0  0 -1  0  0 ] \n
 *    1 [  0 -1 -2 -1  0 ] \n
 *   -- [ -1 -2 16 -2 -1 ] \n
 *   16 [  0 -1 -2 -1  0 ] \n
 *      [  0  0 -1  0  0 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_LAPLACIAN_5, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

public:
    //! Number used to divide filter response
    static const int DENOM = 16;

    //! Filter kernel
    static const CKernel2D::tData KERNEL[];

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 5; }
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsLaplacian.hxx"


} // namespace img
} // namespace mds

#endif // MDS_IF_LAPLACIAN_H

