//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFilters/mdsDoG.h               \n
 * Section: libImage                            \n
 * Date:    2005/10/07                          \n
 *
 * $Id: mdsDoG.h 90 2006-09-13 12:22:57Z spanel $
 * 
 * Description:
 * - Difference of Gaussians filter.
 */

#ifndef MDS_IF_DOG_H
#define MDS_IF_DOG_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Math/mdsBase.h>

#include "../mdsImageFilter.h"
#include "../ImageFilters/mdsGaussian.h"

#include <cmath>


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Difference of Gaussians image filter.
MDS_DECLARE_ID(IF_DOG);


//==============================================================================
/*!
 * Difference of Gaussians image filter.
 * - Parameter I is a used image type.
 * - Policy N is used for filter response normalization.
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_DOG, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;
    typedef typename base::tNormFunc tNormFunc;

public:
    //! Constructor that creates DoG filter.
    CImageFilter(double dSigma1 = 1.0, double dSigma2 = 1.5)
        : m_Gaussian1(dSigma1)
        , m_Gaussian2(dSigma2)
    {}

    //! Destructor
    ~CImageFilter() {}


    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    tResult getResponse(const I& SrcImage, tSize x, tSize y)
    {
        // Compute filter response
        return m_Gaussian2.getResponse(SrcImage, x, y) - m_Gaussian1.getResponse(SrcImage, x, y);
    }

    //! Returns filter standard deviations.
    inline void getSigmas(double& dSigma1, double& dSigma2) const;

    //! Sets the standard deviations.
    inline void setSigmas(double dSigma1, double dSigma2);

protected:
    //! Gaussian filters.
    CImageFilter<tImage, IF_GAUSSIAN> m_Gaussian1, m_Gaussian2;
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsDoG.hxx"


} // namespace img
} // namespace mds

#endif // MDS_IF_DOG_H

