//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFilters/mdsConvolution.h       \n
 * Section: libImage                            \n
 * Date:    2005/09/08                          \n
 *
 * $Id: mdsConvolution.h 90 2006-09-13 12:22:57Z spanel $
 * 
 * Description:
 * - Convolution image filter.
 */

#ifndef MDS_IF_CONVOLUTION_H
#define MDS_IF_CONVOLUTION_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Math/mdsBase.h>

#include "../mdsImageFilter.h"
#include "../mdsImageFunctions.h"
#include "../mdsKernel.h"


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Basic convolution image filter
MDS_DECLARE_ID(IF_CONVOLUTION);


//==============================================================================
/*!
 * General convolution filter.
 * - Parameter I is a used image type
 * - Policy N is used for filter response normalization.
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_CONVOLUTION, N>
    : public CImageFilterBase<I,N>
    , public CKernel2D
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

public:
    //! Constructor that creates a new 2D convolution kernel
    //! - Parameter 'Size' must be an odd number
    CImageFilter(tSize Size) : CKernel2D(Size) {}

    //! Constructor
    CImageFilter(tSize Size, const tData& Value)
        : CKernel2D(Size, Value)
    {}

    //! Constructor
    CImageFilter(tConstDataPtr pData, tSize Size)
        : CKernel2D(pData, Size)
    {}

    //! Constructor
    //! - filter is based on a given convolution kernel
    CImageFilter(const CKernel2D& k) : CKernel2D(k) {}

    //! Destructor
    ~CImageFilter() {}


    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    tResult getResponse(const I& SrcImage, tSize x, tSize y)
    {
        // Compute filter response
        return convolve<tResult>(SrcImage, x, y, *this);
    }
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsConvolution.hxx"


} // namespace img
} // namespace mds

#endif // MDS_IF_CONVOLUTION_H

