//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFilters/mdsAveraging.h         \n
 * Section: libImage                            \n
 * Date:    2005/09/15                          \n
 *
 * $Id: mdsAveraging.h 90 2006-09-13 12:22:57Z spanel $
 * 
 * Description:
 * - Averaging image filters.
 */

#ifndef MDS_IF_AVERAGING_H
#define MDS_IF_AVERAGING_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Math/mdsBase.h>

#include "../mdsImageFilter.h"
#include "../mdsImageFunctions.h"
#include "../mdsKernel.h"


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Averaging image filters
MDS_DECLARE_ID(IF_AVG_3);
MDS_DECLARE_ID(IF_AVG_5);
MDS_DECLARE_ID(IF_AVG_7);


//==============================================================================
/*!
 * Averaging image filter (3x3 version).
 * - Parameter I is a used image type.
 * - Policy N is used for filter response normalization.
 * - Kernel definition: \n
 *    1 [ 1  2  1 ] \n
 *   -- [ 2  4  2 ] \n
 *   16 [ 1  2  1 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_AVG_3, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

public:
    //! Number used to divide filter response.
    static const int DENOM = 16;

    //! Filter kernel.
    static const CKernel2D::tData KERNEL[];

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 3; }
};


//==============================================================================
/*!
 * Averaging image filter (5x5 version).
 * - Parameter I is a used image type.
 * - Policy N is used for filter response normalization.
 * - Kernel definition: \n
 *      [ 1  2  3  2  1 ] \n
 *    1 [ 2  4  6  4  2 ] \n
 *   -- [ 3  6  9  6  3 ] \n
 *   81 [ 2  4  6  4  2 ] \n
 *      [ 1  2  3  2  1 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_AVG_5, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

public:
    //! Number used to divide filter response
    static const int DENOM = 81;

    //! Filter kernel
    static const CKernel2D::tData KERNEL[];

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 5; }
};


//==============================================================================
/*!
 * Averaging image filter (7x7 version).
 * - Parameter I is a used image type.
 * - Policy N is used for filter response normalization.
 * - Kernel definition: \n
 *      [ 1  2  3  4  3  2  1 ] \n
 *      [ 2  4  6  8  6  4  2 ] \n
 *   1  [ 3  6  9 12  9  6  3 ] \n
 *  --- [ 4  8 12 16 12  8  4 ] \n
 *  256 [ 3  6  9 12  9  6  3 ] \n
 *      [ 2  4  6  8  6  4  2 ] \n
 *      [ 1  2  3  4  3  2  1 ] \n
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_AVG_7, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

public:
    //! Number used to divide filter response
    static const int DENOM = 256;

    //! Filter kernel
    static const CKernel2D::tData KERNEL[];

public:
    //! Default constructor.
    CImageFilter() {}

    //! Destructor.
    ~CImageFilter() {}

    //! Filtering of input/source image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Value is not normalized!
    tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns filter standard deviation.
    tSize getSize() const { return 7; }
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsAveraging.hxx"


} // namespace img
} // namespace mds

#endif // MDS_IF_AVERAGING_H

