//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFilters/mdsAnisotropic.h       \n
 * Section: libImage                            \n
 * Date:    2007/04/20                          \n
 *
 * $Id:$
 *
 * Description:
 * - Anisotropic filtering based on the diffusion process.
 */


//==============================================================================
/*
 * Methods templates.
 */

// Integration constant chosen according to the number of neighbours
template <class I, template <typename> class N>
//const double CImageFilter<I, IF_ANISOTROPIC, N>::DT          = 0.2;
const double CImageFilter<I, IF_ANISOTROPIC, N>::DT          = 0.125;

// Minimal allowed change of the 
template <class I, template <typename> class N>
//const double CImageFilter<I, IF_ANISOTROPIC, N>::MIN_CHANGE  = 0.01;
const double CImageFilter<I, IF_ANISOTROPIC, N>::MIN_CHANGE  = 0.1;


template <class I, template <typename> class N>
inline double CImageFilter<I, IF_ANISOTROPIC, N>::getDiffusionStrength(double dGrad)
{
    // Magnitude of the image gradient
    double dMagnitude = mds::math::getAbs(dGrad);
    
    // Evaluate the diffusion function
    double dTemp = m_dInvKappa * dMagnitude;
    return std::exp(-dTemp * dTemp);
}


template <class I, template <typename> class N>
inline typename CImageFilter<I, IF_ANISOTROPIC, N>::tResult CImageFilter<I, IF_ANISOTROPIC, N>::getResponse(const I& SrcImage, tSize x, tSize y)
{
    MDS_THROW_ERROR("CImageFilter<I,IF_ANISOTROPIC,N>::getResponse(): Function not implemented");

    return tResult(0);
}


// Image filtering method
template <class I, template <typename> class N>
bool CImageFilter<I, IF_ANISOTROPIC, N>::operator()(const I& SrcImage, I& DstImage)
{
    static const double INV_SQRT2 = 1.0 / mds::math::SQRT2;
    
    // Image size
    tSize XCount = mds::math::getMin(SrcImage.getXSize(), DstImage.getXSize());
    tSize YCount = mds::math::getMin(SrcImage.getYSize(), DstImage.getYSize());
    
    // Check the image margin
    tSize Margin = SrcImage.getMargin();
    if( Margin < 1 )
    {
        return false;
    }
    
    // Copy the source image
    DstImage = SrcImage;

    // Helper image
    CFImage FlowImage(XCount, YCount);

    // Initial value of the flow
    double dFlow = 1.0;

    // Diffusion process
    tSize x, y;
    for( tSize iter = 0; ; ++iter )
    {
        // Check the number of iterations
        if( m_NumOfIters > 0 && iter >= m_NumOfIters )
        {
            break;
        }
        
        // Clear the flow image
        FlowImage.fill(0);
        
        // Filter the image
        for( y = 0; y < YCount; ++y )
        {
            for( x = 0; x < XCount; ++x )
            {
                // Evaluate image derivatives
                double dGradDown = double(DstImage(x,y+1)) - double(DstImage(x,y));
                double dGradTop = double(DstImage(x,y)) - double(DstImage(x,y-1));
                double dGradRight = double(DstImage(x+1,y)) - double(DstImage(x,y)); 
                double dGradLeft = double(DstImage(x,y)) - double(DstImage(x-1,y));

                double dGradTL = INV_SQRT2 * (double(DstImage(x,y)) - double(DstImage(x-1,y-1))); 
                double dGradDR = INV_SQRT2 * (double(DstImage(x+1,y+1)) - double(DstImage(x,y))); 
                double dGradTR = INV_SQRT2 * (double(DstImage(x+1,y-1)) - double(DstImage(x,y))); 
                double dGradDL = INV_SQRT2 * (double(DstImage(x,y)) - double(DstImage(x-1,y+1))); 

                // Evaluate the diffusion function
                double dFlowDown = getDiffusionStrength(dGradDown) * dGradDown;  
                double dFlowTop = getDiffusionStrength(dGradTop) * dGradTop;
                double dFlowRight = getDiffusionStrength(dGradRight) * dGradRight;  
                double dFlowLeft = getDiffusionStrength(dGradLeft) * dGradLeft;  
                
                double dFlowTL = getDiffusionStrength(dGradTL) * dGradTL;  
                double dFlowDR = getDiffusionStrength(dGradDR) * dGradDR;
                double dFlowTR = getDiffusionStrength(dGradTR) * dGradTR;  
                double dFlowDL = getDiffusionStrength(dGradDL) * dGradDL;  

                // Calculate the flow
//                FlowImage(x,y) = tFloatPixel(DT * (dFlowRight - dFlowLeft + dFlowDown - dFlowTop));
                double dDiff = double(SrcImage(x,y)) - double(DstImage(x,y));
//                FlowImage(x,y) = tFloatPixel(DT * (dFlowRight - dFlowLeft + dFlowDown - dFlowTop + dDiff));
                FlowImage(x,y) = tFloatPixel(DT * (dFlowRight - dFlowLeft + dFlowDown - dFlowTop + dFlowDR - dFlowTL + dFlowTR - dFlowDL + dDiff));
            }
        }
        
        // Modify the image and calculate the convergence criterion
        double dNewFlow = 0.0;
        for( y = 0; y < YCount; ++y )
        {
            for( x = 0; x < XCount; ++x )
            {
                DstImage(x,y) = tPixel(tFloatPixel(DstImage(x,y)) + FlowImage(x,y));
                dNewFlow += mds::math::getAbs(FlowImage(x,y));
            }
        }

        // Estimate change of the image
        double dDelta = mds::math::getAbs(dNewFlow / dFlow - 1.0);
        if( dDelta < MIN_CHANGE )
        {
            break;
        }
        dFlow = dNewFlow;
    }

    // O.K.
    return true;
}

