//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageFilters/mdsAnisotropic.h       \n
 * Section: libImage                            \n
 * Date:    2007/04/20                          \n
 *
 * $Id:$
 *
 * Description:
 * - Anisotropic filtering based on the diffusion process.
 */

#ifndef MDS_IF_ANISOTROPIC_H
#define MDS_IF_ANISOTROPIC_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Base/mdsError.h>
#include <MDSTk/Math/mdsBase.h>

#include "../mdsImageFilter.h"

#include <cmath>


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Anisotropic image filter
MDS_DECLARE_ID(IF_ANISOTROPIC);


//==============================================================================
/*!
 * Anisotropic filtering based on the diffusion process.
 * - Parameter I is a used image type.
 * - Policy N is used for normalization.
 */
template <class I, template <typename> class N>
class CImageFilter<I, IF_ANISOTROPIC, N> : public CImageFilterBase<I,N>
{
public:
    //! Image filter base.
    typedef CImageFilterBase<I,N> base;
    typedef typename base::norm norm;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;

    // Integration constant chosen according to the number of neighbours.
    static const double DT;

    // Minimal change of the flow during the iterative diffusion. 
    static const double MIN_CHANGE;

public:
    //! Constructor.
    CImageFilter(double dKappa, tSize NumOfIters = -1)
        : m_dKappa(dKappa)
        , m_NumOfIters(NumOfIters)
    {
        MDS_ASSERT(dKappa > 0.0);
        
        m_dInvKappa = (m_dKappa > 0.0) ? 1.0 / m_dKappa : 1.0;
    }

    //! Destructor
    ~CImageFilter() {}


    //! Filtering of input/source image. Point filter responses are written
    //! to the destination image.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns filter response at specified image point.
    //! - Not implemmented!
    inline tResult getResponse(const I& SrcImage, tSize x, tSize y);

    //! Returns number of iterations of diffusion process.
    tSize getNumOfIters() const { return m_NumOfIters; }

    //! Sets the number of iterations of diffusion process.
    void setNumOfIters(tSize NumOfIters) { m_NumOfIters = NumOfIters; }

protected:
    //! Filter parameters.
    double m_dKappa, m_dInvKappa;

    //! Number of iterations of the diffusion process.
    tSize m_NumOfIters;

protected:
    //! Returns value of the diffusion function.
    inline double getDiffusionStrength(double dGrad);
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsAnisotropic.hxx"


} // namespace img
} // namespace mds

#endif // MDS_IF_ANISOTROPIC_H

