//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)        \n
 * Copyright (c) 2003-2005 by Michal Spanel         \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz      \n
 * File:    ImageEdgeDetectors/mdsZeroCrossings.h   \n
 * Section: libImage                                \n
 * Date:    2005/10/07                              \n
 *
 * $Id: mdsZeroCrossings.h 90 2006-09-13 12:22:57Z spanel $
 * 
 * Description:
 * - Zero crossings of the LoG filter.
 */

#ifndef MDS_IED_ZEROCROSSINGS_H
#define MDS_IED_ZEROCROSSINGS_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Math/mdsBase.h>

#include "../mdsPixelTraits.h"
#include "../mdsImageEdgeDetector.h"
#include "../mdsImageFilters.h"

#include <cmath>


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Zero crossings of the LoG filter.
MDS_DECLARE_ID(IED_ZERO_CROSS);


//==============================================================================
/*!
 * Zero crossings of the LoG filter.
 * - Parameter I is a used image type.
 */
template <class I>
class CImageEdgeDetector<I, IED_ZERO_CROSS> : public CImageEdgeDetectorBase<I>
{
public:
    //! Image edge detector base.
    typedef CImageEdgeDetectorBase<I> base;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;
    typedef typename base::tImageIterator tImageIterator;

public:
    //! Default constructor.
    //! - Standard deviation ("sigma" parameter) of the LoG filter.
    //! - Threshold for the gradient magnitude.
    CImageEdgeDetector(double dSigma, double dThreshold)
        : m_Filter(dSigma)
        , m_dThreshold(dThreshold)
    {
        MDS_ASSERT(dThreshold <= 1.0 && dThreshold >= 0.0);
    }

    //! Destructor.
    ~CImageEdgeDetector() {}


    //! Edge detection in a given image.
    //! - Returns false on failure
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns LoG filter standard deviation.
    double getSigma() const { return m_Filter.getSigma(); }

    //! Sets the LoG filter standard deviation.
    void setSigma(double dSigma) { m_Filter.setSigma(dSigma); }

    //! Return threshold for the gradient magnitude.
    double getThreshold() { return m_dThreshold; }

    //! Returns gradient magnitude threshold.
    inline void setThreshold(double dThreshold);

protected:
    //! LoG filter.
    CImageFilter<tImage, IF_LOG, IFN_MEAN> m_Filter;

    //! Magnitude threshold.
    double m_dThreshold;

protected:
    //! Checks if the pixel is on edge.
    inline bool isEdge(const tImage& Image, tSize x, tSize y);

    //! Computes the gradient magnitude.
    inline double computeGradient(const tImage& Image, tSize x, tSize y);
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsZeroCrossings.hxx"


} // namespace img
} // namespace mds

#endif // MDS_IED_ZEROCROSSINGS_H

