//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageEdgeDetectors/mdsCanny.hxx     \n
 * Section: libImage                            \n
 * Date:    2005/09/20                          \n
 *
 * $Id: mdsCanny.hxx 445 2007-08-24 13:06:46Z spanel $
 *
 * Description:
 * - Canny edge detector.
 */


//==============================================================================
/*
 * Methods of the class mds::img::CImageEdgeDetector<I, IED_CANNY>.
 */

// Canny edge detector
template <class I>
bool CImageEdgeDetector<I, IED_CANNY>::operator()(const I& SrcImage, I& DstImage)
{
    static const float fGray = CPixelTraits<tPixel>::getGray();

    // Image size
    tSize XCount = mds::math::getMin(SrcImage.getXSize(), DstImage.getXSize());
    tSize YCount = mds::math::getMin(SrcImage.getYSize(), DstImage.getYSize());
    tSize Margin = SrcImage.getMargin();

    // Check the margin
    MDS_ASSERT(Margin >= 1);

    // Gaussian smoothing of the input image
    tImage SmoothedImage(XCount, YCount, Margin);
    m_GaussFilter(SrcImage, SmoothedImage);

    // Mirror the image margin to avoid artifacts
    SmoothedImage.mirrorMargin();

    // Compute gradient in x-direction
    tImage GradImageX(XCount, YCount, Margin);
    m_SobelX(SmoothedImage, GradImageX);

    // Compute gradient in y-direction
    tImage GradImageY(XCount, YCount, Margin);
    m_SobelY(SmoothedImage, GradImageY);

    // Compute the magnitude image
    tImage MagnitudeImage(XCount, YCount, Margin);
    for( tSize y = 0; y < YCount; ++y )
    {
        for( tSize x = 0; x < XCount; ++x )
        {
            float fValue = mds::math::getAbs(GradImageX(x, y) - fGray);
            fValue += mds::math::getAbs(GradImageY(x, y) - fGray);
            MagnitudeImage(x, y) = tPixel(0.5f * fValue);
        }
    }

    // Non-maximal suppression
    tImage NonMaxMagnitudeImage(XCount, YCount, Margin);
    nonMaxSuppression(GradImageX,
                      GradImageY,
                      MagnitudeImage,
                      NonMaxMagnitudeImage,
                      XCount, YCount
                      );

    // Thresholds for the hystersis
    // TODO: Estimate both thresholds from image histogram!
    tPixel MaxMagnitude = getMax<tPixel>(NonMaxMagnitudeImage);
    tPixel T1 = tPixel(m_dT1 * MaxMagnitude);
    tPixel T2 = tPixel(m_dT2 * MaxMagnitude);

    // Hysteresis
    DstImage.fillEntire(CPixelTraits<tPixel>::getPixelMin());
    hysteresis(NonMaxMagnitudeImage, DstImage, XCount, YCount, T1, T2);

    // O.K.
    return true;
}


template <class I>
inline void CImageEdgeDetector<I, IED_CANNY>::getThresholds(double& dT1, double& dT2)
{
    dT1 = m_dT1;
    dT2 = m_dT2;
}


template <class I>
inline void CImageEdgeDetector<I, IED_CANNY>::setThresholds(double dT1, double dT2)
{
    MDS_ASSERT(dT1 <= 1.0 && dT2 <= dT1 && dT2 >= 0.0);

    m_dT1 = dT1;
    m_dT2 = dT2;
}


template <class I>
void CImageEdgeDetector<I, IED_CANNY>::nonMaxSuppression(const tImage& GradImageX,
                                                         const tImage& GradImageY,
                                                         tImage& MagnitudeImage,
                                                         tImage& Image,
                                                         tSize XSize,
                                                         tSize YSize
                                                         )
{

    static const tPixel Zero = CPixelTraits<tPixel>::getPixelMin();
    static const tCoordinate Gray = CPixelTraits<tPixel>::getGray();

    // For each pixel
    for( tSize y = 0; y < YSize; ++y )
    {
        for( tSize x = 0; x < XSize; ++x )
        {
            // Check the pixel value
            tPixel Value = MagnitudeImage.get(x, y);
            if( Value > Zero )
            {
                // Derivatives in x and y direction
                tCoordinate cx = GradImageX.get(x, y) - Gray;
                tCoordinate cy = GradImageY.get(x, y) - Gray;

                // Estimate orientation of the edge
                CVector3D Orient(cx, cy);
                Orient.normalize();
                Orient *= 0.75;
                if( mds::math::getAbs(Orient.x) < 0.01 && mds::math::getAbs(Orient.y) < 0.01 )
                {
                    continue;
                }

                // Pixel neighbours
                CPoint3D Left(x + Orient.x, y + Orient.y);
                CPoint3D Right(x - Orient.x, y - Orient.y);

                // Interpolate pixel value in place of neighbours
                tPixel LeftValue = MagnitudeImage.interpolate(Left);
                tPixel RightValue = MagnitudeImage.interpolate(Right);

                // Check if the pixel is local maximum
                if( Value > LeftValue && Value > RightValue )
                {
                    Image.set(x, y, Value);
                }
            }
        }
    }
}


template <class I>
void CImageEdgeDetector<I, IED_CANNY>::hysteresis(tImage& MagnitudeImage,
                                                  tImage& Image,
                                                  tSize XSize,
                                                  tSize YSize,
                                                  tPixel T1,
                                                  tPixel T2
                                                  )
{
    // Iterator pointing after the last pixel of the magnitude image
    const tImageIterator itEnd = MagnitudeImage.getEnd();

    // Accept all pixels whose magnitude exceeds the T1 threshold
    tImageIterator it = MagnitudeImage.getBegin();
    while( it != itEnd )
    {
        if( *it >= T1 )
        {
            Image.set(it.getX(), it.getY(), *it);
        }
        ++it;
    }

    // The number of new edge pixels
    int iCount;

    // Until stability is achieved
    do {
        // Reset the number of new edge pixels
        iCount = 0;

        // For each pixel
        tImageIterator it = MagnitudeImage.getBegin();
        while( it != itEnd )
        {
            // Check the pixel value
            tPixel Value = *it;
            if( Value >= T2 && Value < T1 )
            {
                // Get the pixel position
                tSize x = it.getX();
                tSize y = it.getY();

                // Check the neighbours
                if( checkNeighbours(MagnitudeImage, x, y, T1) )
                {
                    Image.set(x, y, *it);
                    *it = T1;
                    ++iCount;
                }
            }

            // Next pixel
            ++it;
        }
    } while( iCount > 0 );
}


template <class I>
bool CImageEdgeDetector<I, IED_CANNY>::checkNeighbours(tImage& Image,
                                                       tSize x,
                                                       tSize y,
                                                       tPixel T
                                                       )
{
    // Volume properties
    tSize XOffset = Image.getXOffset();
    tSize YOffset = Image.getYOffset();

    // Pointer to the pixel
    tPixel *p = Image.getPtr(x, y);

    // Check the neighbours
    return (*(p + YOffset) >= T
            || *(p - YOffset) >= T
            || *(p + XOffset) >= T
            || *(p + XOffset + YOffset) >= T
            || *(p + XOffset - YOffset) >= T
            || *(p - XOffset) >= T
            || *(p - XOffset + YOffset) >= T
            || *(p - XOffset - YOffset) >= T
            );
}

