//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    ImageEdgeDetectors/mdsCanny.h       \n
 * Section: libImage                            \n
 * Date:    2005/09/20                          \n
 *
 * $Id: mdsCanny.h 90 2006-09-13 12:22:57Z spanel $
 * 
 * Description:
 * - Canny edge detector.
 */

#ifndef MDS_IED_CANNY_H
#define MDS_IED_CANNY_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Math/mdsBase.h>

#include "../mdsPixelTraits.h"
#include "../mdsImageEdgeDetector.h"
#include "../mdsImageFilters.h"
#include "../mdsVector.h"

#include <cmath>


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Canny edge detector
MDS_DECLARE_ID(IED_CANNY);


//==============================================================================
/*!
 * Canny edge detector.
 * - Parameter I is a used image type.
 */
template <class I>
class CImageEdgeDetector<I, IED_CANNY> : public CImageEdgeDetectorBase<I>
{
public:
    //! Image edge detector base.
    typedef CImageEdgeDetectorBase<I> base;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;
    typedef typename base::tImageIterator tImageIterator;

public:
    //! Default constructor.
    //! - Standard deviation ("sigma" parameter) of the Gaussian filter
    //!   which is used for the smoothing.
    //! - Thresholds for the hysteresis (0..1).
    CImageEdgeDetector(double dSigma, double dT1, double dT2)
        : m_GaussFilter(dSigma)
        , m_dT1(dT1)
        , m_dT2(dT2)
    {
        MDS_ASSERT(dT1 <= 1.0 && dT2 <= dT1 && dT2 >= 0.0);
    }

    //! Destructor.
    ~CImageEdgeDetector() {}


    //! Edge detection in a given image.
    //! - Magnitude image is produced.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns smoothing filter standard deviation.
    double getSigma() const { return m_GaussFilter.getSigma(); }

    //! Sets the smoothing filter standard deviation.
    void setSigma(double dSigma) { m_GaussFilter.setSigma(dSigma); }

    //! Returns thresholds of the non-maximal suppression algorithm.
    inline void getThresholds(double& dT1, double& dT2);

    //! Sets thresholds of the non-maximal suppression algorithm.
    inline void setThresholds(double dT1, double dT2);

protected:
    //! Gaussian smoothing filter.
    CImageFilter<tImage, IF_GAUSSIAN> m_GaussFilter;

    //! Sobel operators.
    CImageFilter<tImage, IF_SOBEL_X, IFN_MEAN> m_SobelX;
    CImageFilter<tImage, IF_SOBEL_Y, IFN_MEAN> m_SobelY;

    //! Thresholds used by hysteresis algorithm.
    double m_dT1, m_dT2;

protected:
    //! Non-maximal suppression.
    void nonMaxSuppression(const tImage& GradImageX,
                           const tImage& GradImageY,
                           tImage& MagnitudeImage,
                           tImage& Image,
                           tSize XSize,
                           tSize YSize
                           );

    //! Hystersis controlled by two given thresholds.
    //! - Applies the hysteresis to the provided slice. It checks each pixel
    //!   against an upper threshhold 'T1'. If the pixel value is equal
    //!   or higher, the current pixel is an edge pixel. Then all direct
    //!   neighbours of the pixel are compared to a lower threshhold 'T2'
    //!   and if their values are equal or higher than this threshhold,
    //!   they become edge pixels as well.
    //! - In practice the threshold T1 = (2-3) * T2.
    void hysteresis(tImage& MagnitudeImage,
                    tImage& Image,
                    tSize XSize,
                    tSize YSize,
                    tPixel T1,
                    tPixel T2
                    );

    //! Returns true if any neighbouring pixel has greater value
    //! than a given threshold.
    bool checkNeighbours(tImage& Image, tSize x, tSize y, tPixel T);
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsCanny.hxx"


} // namespace img
} // namespace mds

#endif // MDS_IED_CANNY_H

