//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)       \n
 * Copyright (c) 2003-2007 by PGMed@FIT            \n
 *
 * Author:  Petr Hyna, xhynap00@stud.fit.vutbr.cz  \n
 *          (modified by Michal Spanel)            \n
 * File:    ImageCornerDetectors/mdsHarris.h       \n
 * Section: libImage                               \n
 * Date:    2007/04/12                             \n
 *
 * $Id:$
 *
 * Description:
 * - Harris corner detector.
 */

#ifndef MDS_ICD_HARRIS_H
#define MDS_ICD_HARRIS_H

#include <MDSTk/Base/mdsSetup.h>
#include <MDSTk/Base/mdsAssert.h>
#include <MDSTk/Math/mdsBase.h>

#include "../mdsPixelTraits.h"
#include "../mdsImageCornerDetector.h"
#include "../mdsImageFilters.h"
#include "../mdsVector.h"

#include <cmath>


namespace mds
{
namespace img
{

//==============================================================================
/*
 * Identifiers declarations.
 */

//! Harris corner detector
MDS_DECLARE_ID(ICD_HARRIS);


//==============================================================================
/*!
 * Harris corner detector.
 * - It is better to smoooth input image using the Gaussian convolution filter
 *   beforehand.
 * - Parameter I is a used image type.
 */
template <class I>
class CImageCornerDetector<I, ICD_HARRIS> : public CImageCornerDetectorBase<I>
{
public:
    //! Image corner detector base.
    typedef CImageCornerDetectorBase<I> base;
    typedef typename base::tImage tImage;
    typedef typename base::tPixel tPixel;
    typedef typename base::tResult tResult;
    typedef typename base::tImageIterator tImageIterator;

public:
    //! Constructor.
    //! - Standard deviation of gaussian window used to estimate eigen values.
    //! - Cornerness threshold.
    //! - Tuneable parameter.
    CImageCornerDetector(double dSigma, double dThreshold, double dKappa)
        : m_GaussFilter(dSigma)
        , m_dThreshold(dThreshold)
		, m_dKappa(dKappa)
    {
        MDS_ASSERT(dThreshold <= 1000 && dKappa <= 0.25 && dKappa >= 0.01);
    }

    //! Destructor.
    ~CImageCornerDetector() {}
 

    //! Corner detection in a given image.
    //! - Magnitude image is produced.
    //! - Returns false on failure.
    bool operator()(const I& SrcImage, I& DstImage);

    //! Returns threshold of the non-maximal suppression algorithm.
    double getThreshold() { return m_dThreshold; }

    //! Sets threshold of the non-maximal suppression algorithm.
    inline void setThreshold(double dThreshold);

	//! Returns width of the filter for getting corners.
    double getKappa() { return m_dKappa; }

    //! Sets width of the filter for gettig corners.
    inline void setKappa(double dKappa);

protected:
    //! Gaussian filter.
    CImageFilter<CFImage, IF_GAUSSIAN> m_GaussFilter;    
    CImageFilter<CFImage, IF_SOBEL_X> m_SobelX;
    CImageFilter<CFImage, IF_SOBEL_Y> m_SobelY;

    //! Harris corner detector parameters.
	double m_dThreshold, m_dKappa;

protected:
    //! Returns true if any neighbouring pixel has greater value
    //! than a given threshold.
    bool checkNeighbours(CFImage& Image, tSize x, tSize y, tFloatPixel T);
};


//==============================================================================
/*
 * Methods templates.
 */

// Include file containing methods templates
#include "mdsHarris.hxx"


} // namespace img
} // namespace mds

#endif // MDS_ICD_HARRIS_H

