//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsStaticData.h                     \n
 * Section: libBase                             \n
 * Date:    2006/02/15                          \n
 *
 * $Id: mdsStaticData.h 64 2006-08-11 08:45:24Z spanel $
 * 
 * Description:
 * - Template providing data allocation functions.
 * - Implemented via static array.
 */

#ifndef MDS_STATICDATA_H
#define MDS_STATICDATA_H

#include "mdsSetup.h"
#include "mdsAssert.h"
#include "mdsTypes.h"
#include "mdsMemory.h"


namespace mds
{
namespace base
{

//==============================================================================
/*!
 * Template providing data allocation functions.
 * - Implemented via static array.
 */
template <typename T, tSize M>
class CStaticData
{
public:
    //! Data type.
    typedef T tData;

    //! Pointer to the data.
    typedef T *tDataPtr;

    //! Pointer to the constant data.
    typedef const T *tConstDataPtr;

public:
    //! Default constructor.
    inline CStaticData();

    //! Constructor initializes the data.
    inline CStaticData(const tData& Value);

    //! Constructor that copies a specified data.
    inline CStaticData(tConstDataPtr pData, tSize Size);

    //! Constructor copies a given data.
    inline CStaticData(const CStaticData& Data);

    //! Destructor.
    inline ~CStaticData();

    //! Assignment operator.
    inline CStaticData& operator =(const CStaticData& Data);

    //! Initializes the data.
    inline void create(const tData& Value);

    //! Copies the specified data.
    inline void create(tConstDataPtr pData, tSize Size);

    //! Allocates memory and copies the specified data.
    inline void create(const CStaticData& Data);


    //! Returns data size.
    tSize getSize() const { return M; }

    //! Returns the subscripted element [i].
    T& operator ()(tSize i)
    {
        return m_pData[i];
    }
    const T& operator ()(tSize i) const
    {
        return m_pData[i];
    }

    //! Gets the element [i].
    T& get(tSize i)
    {
        return m_pData[i];
    }
    const T& get(tSize i) const
        {
            return m_pData[i];
        }

    //! Returns pointer to the given element.
    T *getPtr(tSize i)
    {
        return (m_pData + i);
    }
    const T *getPtr(tSize i) const
    {
        return (m_pData + i);
    }

    //! Sets the element of matrix at the position [i].
    void set(tSize i, const T& Value)
    {
        m_pData[i] = Value;
    }

    //! Returns pointer to the data.
    T *getPtr()
    {
        return m_pData;
    }
    const T *getPtr() const
    {
        return m_pData;
    }

protected:
    //! Statically allocated data.
    tData m_pData[M];
};


//==============================================================================
/*
 * Method templates
 */

// Include file containing method templates
#include "mdsStaticData.hxx"


} // namespace base
} // namespace mds

#endif // MDS_STATICDATA_H

