//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsSharedPtr.h                      \n
 * Section: libBase                             \n
 * Date:    2004/10/17                          \n
 *
 * $Id: mdsSharedPtr.h 375 2007-06-19 14:42:03Z spanel $
 *
 * Description:
 * - Smart pointer to a shared object.
 * - Intrusive reference counting.
 */

#ifndef MDS_SHAREDPTR_H
#define MDS_SHAREDPTR_H

#include "mdsSetup.h"
#include "mdsAssert.h"
#include "mdsObject.h"


namespace mds
{
namespace base
{

//==============================================================================
/*
 * Global macros.
 */

//! Macro declares a class member tSmartPtr which can be later used to create
//! smart pointer to class T. It also allows class instantiation by smart
//! pointer only. So it should be located in the private part of the class
//! as well as default constructor, copy constructor and all others
//! constructors.
#define MDS_SHAREDPTR(T) \
    typedef mds::base::CSharedPtr<T> tSmartPtr; \
    friend class mds::base::CSharedPtr<T>


//==============================================================================
/*!
 * Smart pointer providing intrusive reference counting. Once the pointer
 * is initialized, it takes full control over the pointer including deletion.
 */
template <typename T>
class CSharedPtr
{
public:
    //! Check that T is derived from CObject class. Otherwise, you will see
    //! name of this enum somewhere in compiler error message.
    enum { TEMPLATE_PARAMETER_IS_NOT_OBJECT = T::CLASS_OBJECT };

    //! Stored pointer type.
    typedef T *tPointer;

    //! Pointee type.
    typedef T tPointee;

public:
/*    //! Default constructor.
    //! - Internal pointer is initialized to the NULL.
    //! - Use with care!
    CSharedPtr() : m_pInstance(0) {}*/

    //! Default constructor.
    //! - Uses default pointee type constructor.
    CSharedPtr() : m_pInstance(new T())
    {
        MDS_ASSERT(m_pInstance);
        m_pInstance->addReference();
    }

/*    //! Constructor.
    //! - Uses pointee type constructor.
    //! - Obsolete!
    template<typename T1>
    CSharedPtr(const T1& p1) : m_pInstance(new T(p1))
    {
        MDS_ASSERT(m_pInstance);
        m_pInstance->addReference();
    }*/

    //! Constructor from a raw pointer.
    CSharedPtr(T *p) : m_pInstance(p)
    {
        if( m_pInstance )
        {
            m_pInstance->addReference();
        }
    }

    //! Constructor from a raw pointer.
/*    CSharedPtr(const T *p) : m_pInstance(const_cast<T *>(p))
    {
        if( m_pInstance )
        {
            m_pInstance->addReference();
        }
    }*/

    //! Copy constructor.
    //! - Makes a new reference to the object.
    CSharedPtr(const CSharedPtr& p) : m_pInstance(p.m_pInstance)
    {
        m_pInstance->addReference();
    }

/*    //! Constructor.
    //! - Uses pointee type constructor.
    //! - Obsolete!
    template<typename T1, typename T2>
    CSharedPtr(const T1& p1, const T2& p2) : m_pInstance(new T(p1, p2))
    {
        MDS_ASSERT(m_pInstance);
        m_pInstance->addReference();
    }

    //! Constructor.
    //! - Uses pointee type constructor parameters.
    //! - Obsolete!
    template<typename T1, typename T2, typename T3>
    CSharedPtr(const T1& p1, const T2& p2, const T3& p3)
            : m_pInstance(new T(p1, p2, p3))
    {
        MDS_ASSERT(m_pInstance);
        m_pInstance->addReference();
    }

    //! Constructor.
    //! - Uses pointee type constructor parameters.
    //! - Obsolete!
    template<typename T1, typename T2, typename T3, typename T4>
    CSharedPtr(const T1& p1, const T2& p2, const T3& p3, const T4& p4)
            : m_pInstance(new T(p1, p2, p3, p4))
    {
        MDS_ASSERT(m_pInstance);
        m_pInstance->addReference();
    }

    //! Constructor.
    //! - Uses pointee type constructor parameters.
    //! - Obsolete!
    template<typename T1, typename T2, typename T3, typename T4, typename T5>
    CSharedPtr(const T1& p1, const T2& p2, const T3& p3, const T4& p4, const T5& p5)
            : m_pInstance(new T(p1, p2, p3, p4, p5))
    {
        MDS_ASSERT(m_pInstance);
        m_pInstance->addReference();
    }

    //! Constructor.
    //! - Uses pointee type constructor.
    //! - Obsolete!
    template<typename T1, typename T2, typename T3, typename T4, typename T5, typename T6>
    CSharedPtr(const T1& p1, const T2& p2, const T3& p3, const T4& p4, const T5& p5, const T6& p6)
            : m_pInstance(new T(p1, p2, p3, p4, p5, p6))
    {
        MDS_ASSERT(m_pInstance);
        m_pInstance->addReference();
    }*/


    //! Assignment operator.
    //! - Makes a new reference to the object.
    CSharedPtr& operator=(const CSharedPtr& p)
    {
        if( this != &p )
        {
            if( m_pInstance->delReference() )
            {
                delete m_pInstance;
            }
            m_pInstance = p.m_pInstance;
            m_pInstance->addReference();
        }
        return *this;
    }

    //! Assignment operator.
    //! - Makes a new reference to the object.
    CSharedPtr& operator=(T *p)
    {
        if( m_pInstance != p )
        {
            if( m_pInstance && m_pInstance->delReference() )
            {
                delete m_pInstance;
            }
            m_pInstance = p;
            if( m_pInstance )
            {
                m_pInstance->addReference();
            }
        }
        return *this;
    }

    //! Assignment operator.
    //! - Makes a new reference to the object.
/*    CSharedPtr& operator=(const T *p)
    {
        if( m_pInstance != p )
        {
            if( m_pInstance && m_pInstance->delReference() )
            {
                delete m_pInstance;
            }
            m_pInstance = const_cast<T *>(p);
            if( m_pInstance )
            {
                m_pInstance->addReference();
            }
        }
        return *this;
    }*/

    //! Destructor.
    ~CSharedPtr()
    {
        if( m_pInstance && m_pInstance->delReference() )
        {
            delete m_pInstance;
        }
    }


    //! Conversion to the raw pointer.
    operator T*() { return m_pInstance; }

    //! Added conversion to the void pointer to avoid raw pointer deletion.
    operator void*() { return m_pInstance; }


    //! Returns pointer to the encapsulated object.
    T *operator->() const { return m_pInstance; }

    //! Returns reference to the stored object.
    T& operator*() const { return *m_pInstance; }


    //! Enables "if (!sp) ...".
    bool operator!() const { return (m_pInstance == 0); }


    //! Comparision operators.
    inline friend bool operator==(const CSharedPtr& l, const T* r)
    {
        return (l.m_pInstance == r);
    }

    inline friend bool operator==(const T* l, const CSharedPtr& r)
    {
        return (r.m_pInstance == l);
    }

    inline friend bool operator!=(const CSharedPtr& l, const T* r)
    {
        return (l.m_pInstance != r);
    }

    inline friend bool operator!=(const T* l, const CSharedPtr& r)
    {
        return (r.m_pInstance != l);
    }

    inline friend bool operator<(const CSharedPtr& l, const T* r)
    {
        return (l.m_pInstance < r);
    }

    inline friend bool operator<(const T* l, const CSharedPtr& r)
    {
        return (r.m_pInstance > l);
    }

    //! Ambiquity buster that solves problem using comparision
    //! like "if( p == sp ) ...".
    template<typename U>
    inline friend bool operator==(const CSharedPtr& l, const U* r)
    {
        return (l.m_pInstance == r);
    }

    template<typename U>
    inline friend bool operator==(const U* l, const CSharedPtr& r)
    {
        return (r.m_pInstance == l);
    }

    template<typename U>
    inline friend bool operator!=(const CSharedPtr& l, const U* r)
    {
        return (l.m_pInstance != r);
    }

    template<typename U>
    inline friend bool operator!=(const U* l, const CSharedPtr& r)
    {
        return (r.m_pInstance != l);
    }

    template<typename U>
    inline friend bool operator<(const CSharedPtr& l, const U* r)
    {
        return (l.m_pInstance < r);
    }

    template<typename U>
    inline friend bool operator<(const U* l, const CSharedPtr& r)
    {
        return (r.m_pInstance > l);
    }

    //! Ambiguity buster solving the comparision of two smart pointers.
    template<typename U>
    bool operator==(const CSharedPtr<U>& r) const
    {
        return (m_pInstance == r.m_pInstance);
    }

    template<typename U>
    bool operator!=(const CSharedPtr<U>& r) const
    {
        return (m_pInstance != r.m_pInstance);
    }

    template<typename U>
    bool operator<(const CSharedPtr<U>& r) const
    {
        return (m_pInstance < r.m_pInstance);
    }


    //! Returns pointer to the encapsulated object.
    T *get() { return m_pInstance; }

    //! Returns pointer to the encapsulated object and increases the reference
    //! counter to avoid destruction of the object.
    //! - Use with care!
    T *release()
    {
        if( m_pInstance )
        {
            m_pInstance->addReference();
        }
        return m_pInstance;
    }

private:
    //! Pointer to the encapsulated object.
    T *m_pInstance;
};


} // namespace base
} // namespace mds

#endif // MDS_SHAREDPTR_H

