//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsSTLContainer.h                   \n
 * Section: libBase                             \n
 * Date:    2007/06/10                          \n
 *
 * $Id:$
 *
 * Description:
 * - Container encapsulating any STL container.
 */

#ifndef MDS_STLCONTAINER_H
#define MDS_STLCONTAINER_H

// MDSTk
#include "mdsSetup.h"
#include "mdsIterator.h"
#include "mdsTypeTraits.h"


namespace mds
{
namespace base
{

//==============================================================================
/*!
 * STL iterator wrapper.
 */
template <typename I>
class CSTLIterator : public CSimpleIteratorBase<typename CTypeTraits<typename I::pointer>::tPointee>
{
public:
    //! Iterator base.
    typedef CSimpleIteratorBase<typename CTypeTraits<typename I::pointer>::tPointee> base;

    //! STL iterator.
    typedef I tSTLIterator;

public:
    //! Constructor.
    CSTLIterator(const tSTLIterator& It) : m_STLIterator(It)
    {
        base::m_pItem = &(*m_STLIterator);
    }

    //! Copy constructor.
    CSTLIterator(const CSTLIterator& It) : m_STLIterator(It.m_STLIterator)
    {
        base::m_pItem = It.m_pItem;
    }

    //! Destructor.
    ~CSTLIterator() {}

    //! Assignment operator.
    CSTLIterator& operator=(const CSTLIterator& It)
    {
        m_STLIterator = It.m_STLIterator;
        base::m_pItem = It.m_pItem;
        return *this;
    }

    //! Returns reference to the STL iterator.
    tSTLIterator& getIterator() { return m_STLIterator; }
    const tSTLIterator& getIterator() const { return m_STLIterator; }


    //! Operator moves the iterator to the next pixel.
    //! - Prefix notation.
    CSTLIterator& operator++()
    {
        std::advance(m_STLIterator, 1);
        base::m_pItem = &(*m_STLIterator);
        return *this;
    }

    //! Operator moves the iterator to the next pixel.
    //! - Postfix notation.
    CSTLIterator operator++(int)
    {
        CSTLIterator Temp(*this);
        std::advance(m_STLIterator, 1);
        base::m_pItem = &(*m_STLIterator);
        return Temp;
    }


    //! Returns the number of increments between the positions addressed.
    //! by two iterators.
    tSize getDistance(const CSTLIterator& End) const
    {
        return tSize(std::distance(m_STLIterator, End.m_STLIterator));
    }

protected:
    //! Wrapped STL iterator.
    tSTLIterator m_STLIterator;
};


//==============================================================================
/*!
 * STL container wrapper.
 */
template <typename T>
class CSTLContainer
{
public:
    //! Iterator is declared!
    enum { ITERATOR_DECLARED };
    
    //! STL container.
    typedef T tSTLContainer;
    
    //! Iterator type.
    typedef CSTLIterator<typename T::iterator> tIterator;
    
    //! Const iterator type.
    typedef CSTLIterator<typename T::const_iterator> tConstIterator;

public:
    //! Constructor.
    CSTLContainer(tSTLContainer& Container) : m_STLContainer(Container) {}

    //! Destructor.
    ~CSTLContainer() {}


    //! Returns iterator that points at the first element.
    tIterator getBegin()
    {
        return tIterator(m_STLContainer.begin());
    }
    tConstIterator getBegin() const
    {
        return tConstIterator(m_STLContainer.begin());
    }

    //! Returns iterator that points beyond the last element.
    tIterator getEnd()
    {
        return tIterator(m_STLContainer.end());
    }
    tConstIterator getEnd() const
    {
        return tConstIterator(m_STLContainer.end());
    }

protected:
    //! Reference to the STL container.
    tSTLContainer& m_STLContainer;
};


} // namespace base
} // namespace mds

#endif // MDS_STLCONTAINER_H

