//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsSHMData.h                        \n
 * Section: libBase                             \n
 * Date:    2007/06/21                          \n
 *
 * $Id:$
 *
 * Description:
 * - Template for manipulating references to data array.
 * - Allocation using malloc() and free() functions.
 * - Provides own data reference counting mechanism.
 * - Allows to share data over shared memory.
 */

//==============================================================================
/*
 * Methods templates.
 */

// Default constructor
template <typename T>
inline CSHMData<T>::CSHMData()
{
}


// Constructor that allocates a given size of data
template <typename T>
inline CSHMData<T>::CSHMData(tSize Size)
    : m_pRef(new SRefInfo(Size))
{
    MDS_ASSERT(m_pRef);
}


// Constructor allocates and initializes data
template <typename T>
inline CSHMData<T>::CSHMData(tSize Size, const tData& Value)
    : m_pRef(new SRefInfo(Size))
{
    MDS_ASSERT(m_pRef);

    memSet(m_pRef->m_pData, Value, m_pRef->m_Size);
}


// Constructor that copies a specified data
template <typename T>
inline CSHMData<T>::CSHMData(tConstDataPtr pData, tSize Size)
    : m_pRef(new SRefInfo(Size))
{
    MDS_ASSERT(m_pRef && pData);

    memCopy(m_pRef->m_pData, pData, m_pRef->m_Size);
}


// Constructor copies a given data
template <typename T>
inline CSHMData<T>::CSHMData(const CSHMData<T>& Data)
    : m_pRef(new SRefInfo(Data.m_pRef->m_Size))
{
    MDS_ASSERT(m_pRef);

    memCopy(m_pRef->m_pData, Data.m_pRef->m_pData, m_pRef->m_Size);
}


// Constructor which make a reference to existing data
template <typename T>
inline CSHMData<T>::CSHMData(const CSHMData<T>& Data, EMakeRef MakeRef)
    : m_pRef(Data.m_pRef)
{
    ++m_pRef->m_iReferences;
}


// Virtual destructor
template <typename T>
CSHMData<T>::~CSHMData()
{
    if( (--m_pRef->m_iReferences) == 0 )
    {
        delete m_pRef;
    }
}


// Assignment operator
template <typename T>
inline CSHMData<T>& CSHMData<T>::operator=(const CSHMData<T>& Data)
{
    if( this != &Data )
    {
        create(Data);
    }
    return *this;
}


// Creates new data of a given size
template <typename T>
inline void CSHMData<T>::init(tSize Size)
{
    m_pRef = new SRefInfo(Size);
    MDS_ASSERT(m_pRef);
}


// Creates new data of a given size
template <typename T>
inline void CSHMData<T>::resize(tSize Size)
{
    if( m_pRef->m_iReferences > 1 )
    {
        --m_pRef->m_iReferences;
        m_pRef = new SRefInfo(Size);
    }
    else
    {
        if( Size > m_pRef->m_Size )
        {
            delete m_pRef;
            m_pRef = new SRefInfo(Size);
        }
        else
        {
            m_pRef->m_Size = Size;
        }
    }

    MDS_ASSERT(m_pRef);
}


// Allocates and initializes data
template <typename T>
inline void CSHMData<T>::create(tSize Size, const tData& Value)
{
    resize(Size);

    memSet(m_pRef->m_pData, Value, m_pRef->m_Size);
}


// Creates new data and copies the specified data to it
template <typename T>
inline void CSHMData<T>::create(tConstDataPtr pData, tSize Size)
{
    MDS_ASSERT(pData);

    resize(Size);

    memCopy(m_pRef->m_pData, pData, m_pRef->m_Size);
}


// Creates new data and copies the specified data
template <typename T>
inline void CSHMData<T>::create(const CSHMData<T>& Data)
{
    resize(Data.m_pRef->m_Size);

    memCopy(m_pRef->m_pData, Data.m_pRef->m_pData, m_pRef->m_Size);
}


// Makes a reference to existing data
template <typename T>
inline void CSHMData<T>::create(const CSHMData<T>& Data, EMakeRef)
{
    if( (--m_pRef->m_iReferences) <= 0 )
    {
        delete m_pRef;
    }

    m_pRef = Data.m_pRef;
    ++m_pRef->m_iReferences;
}


template <typename T>
inline void CSHMData<T>::create(mds::sys::CSharedMem *pSharedMem, tDataPtr pBegin, tSize Size)
{
    if( m_pRef->m_iReferences > 1 )
    {
        --m_pRef->m_iReferences;
    }
    else
    {
        delete m_pRef;
    }

    m_pRef = new SRefInfo(pSharedMem, pBegin, Size);
}

