//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsSHMData.h                        \n
 * Section: libBase                             \n
 * Date:    2007/06/21                          \n
 *
 * $Id:$
 *
 * Description:
 * - Template for manipulating references to data array.
 * - Allocation using malloc() and free() functions.
 * - Provides own data reference counting mechanism.
 * - Allows to share data over shared memory.
 */

#ifndef MDS_SHMDATA_H
#define MDS_SHMDATA_H

#include "mdsRefData.h"

#include <MDSTk/System/mdsSharedMem.h>


namespace mds
{

namespace base
{

//==============================================================================
/*!
 * Data allocation with reference counting.
 * - Allows to share data over shared memory. 
 */
template <typename T>
class CSHMData
{
public:
    //! Data type.
    typedef T tData;

    //! Pointer to the data.
    typedef T *tDataPtr;

    //! Pointer to the constant data.
    typedef const T *tConstDataPtr;

public:
    //! Default constructor.
    //! - No allocation of the data is done.
    //! - Be sure that the method init(...) is used somewhere.
    //! - Avoid of using it!
    inline CSHMData();

    //! Constructor that allocates a given size of data.
    //! - Data of the zero size can be allocated.
    inline CSHMData(tSize Size);

    //! Constructor allocates and initializes data.
    inline CSHMData(tSize Size, const tData& Value);

    //! Constructor that copies a specified data.
    inline CSHMData(tConstDataPtr pData, tSize Size);

    //! Constructor copies a given data.
    inline CSHMData(const CSHMData& Data);

    //! Constructor which makes a reference to an existing data.
    //! - Can be called in this way CSHMData(Data, REFERENCE).
    inline CSHMData(const CSHMData& Data, EMakeRef);

    //! Destructor.
    inline ~CSHMData();

    //! Assignment operator.
    inline CSHMData& operator=(const CSHMData& Data);

    //! Allocates data of a given size.
    //! - This method is primarily designed for data reference initialization
    //!   when the default constructor was used.
    //! - Provides no checking of the existing data and its (de,re)allocation.
    //! - Avoid of using it!
    inline void init(tSize Size);

    //! Allocates data of a given size.
    //! - In case of several references to the existing data, the reference
    //!   counter is decremented and new data are allocated.
    inline void resize(tSize Size);

    //! Allocates and initializes data.
    inline void create(tSize Size, const tData& Value);

    //! Allocates data and copies the specified data to it.
    inline void create(tConstDataPtr pData, tSize Size);

    //! Allocates data and copies the specified data.
    inline void create(const CSHMData& Data);

    //! Makes a new reference to existing data.
    //! - Eventually deallocates the associated data.
    inline void create(const CSHMData& Data, EMakeRef);

    //! Constructs data in a shared memory.
    inline void create(mds::sys::CSharedMem *pSharedMem, tDataPtr pBegin, tSize Size);


    //! Returns data size.
    tSize getSize() const { return m_pRef->m_Size; }

    //! Returns the number of references to the data.
    int getNumOfReferences() const { return m_pRef->m_iReferences; }

    //! Returns pointer to the shared memory or NULL if data are not located
    //! in the shared memory.
    mds::sys::CSharedMem *getSharedMemPtr() { return m_pRef->m_spSharedMem; }

    //! Returns the subscripted element [i].
    T& operator ()(tSize i)
    {
        return m_pRef->m_pData[i];
    }
    const T& operator ()(tSize i) const
    {
        return m_pRef->m_pData[i];
    }

    //! Gets the element [i].
    T& get(tSize i)
    {
        return m_pRef->m_pData[i];
    }
    const T& get(tSize i) const
    {
        return m_pRef->m_pData[i];
    }

    //! Returns pointer to the given element.
    T *getPtr(tSize i)
    {
        return (m_pRef->m_pData + i);
    }
    const T *getPtr(tSize i) const
    {
        return (m_pRef->m_pData + i);
    }

    //! Sets the element of matrix at the position [i].
    void set(tSize i, const T& Value)
    {
        m_pRef->m_pData[i] = Value;
    }

    //! Returns pointer to the data.
    tDataPtr getPtr()
    {
        return m_pRef->m_pData;
    }
    const tDataPtr getPtr() const
    {
        return m_pRef->m_pData;
    }

protected:
    //! Structure containing information on allocated data
    //! and the number of references to this data.
    struct SRefInfo /*: public CSmallValueObject<>*/
    {
        //! Size of the allocated data.
        tSize m_Size;

        //! Pointer to the allocated data.
        tDataPtr m_pData;

        //! The number of references to the data.
        //! - Zero if there is no reference.
        int m_iReferences;

        //! Shared memory.
        mds::sys::CSharedMemPtr m_spSharedMem;

        //! Constructor that allocates a given size of data.
        //! - Allocation of the zero size data is allowed.
        SRefInfo(tSize Size)
            : m_Size(Size)
            , m_iReferences(1)
            , m_spSharedMem(NULL)
        {
            MDS_ASSERT(Size >= 0);

            m_pData = memAlloc<tData>(m_Size);
        }

        //! Constructor that allocates a given size of data.
        //! - Allocation of the zero size data is allowed.
        SRefInfo(mds::sys::CSharedMem *pSharedMem, tDataPtr pBegin, tSize Size)
            : m_Size(Size)
            , m_pData(pBegin)
            , m_iReferences(1)
            , m_spSharedMem(pSharedMem)
        {
            MDS_ASSERT(pSharedMem && pBegin && Size >= 0);
        }

        //! Destructor.
        //! - Deallocates the data.
        ~SRefInfo()
        { 
            if( !m_spSharedMem )
            {
                memFree(m_pData);
            }
        }
    };

    //! Pointer to the structure containing the referenced data.
    SRefInfo *m_pRef;
};


//==============================================================================
/*
 * Method templates
 */

// Include file containing method templates
#include "mdsSHMData.hxx"


} // namespace base
} // namespace mds

#endif // MDS_SHMDATA_H

