//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsRefData.hxx                      \n
 * Section: libBase                             \n
 * Date:    2004/11/22                          \n
 *
 * $Id: mdsRefData.hxx 64 2006-08-11 08:45:24Z spanel $
 * 
 * Description:
 * - Template for manipulating references to data array.
 * - Allocation using malloc() and free() functions.
 * - Provides own data reference counting mechanism.
 */

//==============================================================================
/*
 * Methods templates.
 */

// Default constructor
template <typename T>
inline CRefData<T>::CRefData()
{
}


// Constructor that allocates a given size of data
template <typename T>
inline CRefData<T>::CRefData(tSize Size)
        : m_pRef(new SRefInfo(Size))
{
    MDS_ASSERT(m_pRef);
}


// Constructor allocates and initializes data
template <typename T>
inline CRefData<T>::CRefData(tSize Size, const tData& Value)
        : m_pRef(new SRefInfo(Size))
{
    MDS_ASSERT(m_pRef);

    memSet(m_pRef->m_pData, Value, m_pRef->m_Size);
}


// Constructor that copies a specified data
template <typename T>
inline CRefData<T>::CRefData(tConstDataPtr pData, tSize Size)
        : m_pRef(new SRefInfo(Size))
{
    MDS_ASSERT(m_pRef && pData);

    memCopy(m_pRef->m_pData, pData, m_pRef->m_Size);
}


// Constructor copies a given data
template <typename T>
inline CRefData<T>::CRefData(const CRefData<T>& Data)
        : m_pRef(new SRefInfo(Data.m_pRef->m_Size))
{
    MDS_ASSERT(m_pRef);

    memCopy(m_pRef->m_pData, Data.m_pRef->m_pData, m_pRef->m_Size);
}


// Constructor which make a reference to existing data
template <typename T>
inline CRefData<T>::CRefData(const CRefData<T>& Data, EMakeRef MakeRef)
        : m_pRef(Data.m_pRef)
{
    ++m_pRef->m_iReferences;
}


// Virtual destructor
template <typename T>
CRefData<T>::~CRefData()
{
    if( (--m_pRef->m_iReferences) == 0 )
    {
        delete m_pRef;
    }
}


// Assignment operator
template <typename T>
inline CRefData<T>& CRefData<T>::operator=(const CRefData<T>& Data)
{
    if( this != &Data )
    {
        create(Data);
    }
    return *this;
}


// Creates new data of a given size
template <typename T>
inline void CRefData<T>::init(tSize Size)
{
    m_pRef = new SRefInfo(Size);
    MDS_ASSERT(m_pRef);
}


// Creates new data of a given size
template <typename T>
inline void CRefData<T>::resize(tSize Size)
{
    if( m_pRef->m_iReferences > 1 )
    {
        --m_pRef->m_iReferences;
        m_pRef = new SRefInfo(Size);
    }
    else
    {
        if( Size > m_pRef->m_Size )
        {
            delete m_pRef;
            m_pRef = new SRefInfo(Size);
        }
        else
        {
            m_pRef->m_Size = Size;
        }
    }

    MDS_ASSERT(m_pRef);
}


// Allocates and initializes data
template <typename T>
inline void CRefData<T>::create(tSize Size, const tData& Value)
{
    resize(Size);

    memSet(m_pRef->m_pData, Value, m_pRef->m_Size);
}


// Creates new data and copies the specified data to it
template <typename T>
inline void CRefData<T>::create(tConstDataPtr pData, tSize Size)
{
    MDS_ASSERT(pData);

    resize(Size);

    memCopy(m_pRef->m_pData, pData, m_pRef->m_Size);
}


// Creates new data and copies the specified data
template <typename T>
inline void CRefData<T>::create(const CRefData<T>& Data)
{
    resize(Data.m_pRef->m_Size);

    memCopy(m_pRef->m_pData, Data.m_pRef->m_pData, m_pRef->m_Size);
}


// Makes a reference to existing data
template <typename T>
inline void CRefData<T>::create(const CRefData<T>& Data, EMakeRef)
{
    if( (--m_pRef->m_iReferences) <= 0 )
    {
        delete m_pRef;
    }

    m_pRef = Data.m_pRef;
    ++m_pRef->m_iReferences;
}

