//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsDataObject.h                     \n
 * Section: libBase                             \n
 * Date:    2004/11/22                          \n
 *
 * $Id: mdsRefData.h 379 2007-06-20 13:05:57Z spanel $
 *
 * Description:
 * - Template for manipulating references to data array.
 * - Allocation using malloc() and free() functions.
 * - Provides own data reference counting mechanism.
 */

#ifndef MDS_REFDATA_H
#define MDS_REFDATA_H

#include "mdsSetup.h"
#include "mdsAssert.h"
#include "mdsTypes.h"
#include "mdsMemory.h"
//#include "mdsSmallObject.h"


namespace mds
{

//==============================================================================
/*
 * Global constants.
 */

//! Enumeration constant used for making reference to data. It is usually
//! the last parameter of a constructor. So, there are two constructor
//! types. First is used to make a new copy of an object,
//! second to make a new reference.
enum EMakeRef
{
    REFERENCE
};


namespace base
{

//==============================================================================
/*!
 * Template providing data allocation functions and mechanism of references
 * to the allocated data.
 */
template <typename T>
class CRefData
{
public:
    //! Data type.
    typedef T tData;

    //! Pointer to the data.
    typedef T *tDataPtr;

    //! Pointer to the constant data.
    typedef const T *tConstDataPtr;

public:
    //! Default constructor.
    //! - No allocation of the data reference is done.
    //! - Be sure that the method init(...) is used somewhere.
    //! - Avoid of using it!
    inline CRefData();

    //! Constructor that allocates a given size of data.
    //! - Data of the zero size can be allocated.
    inline CRefData(tSize Size);

    //! Constructor allocates and initializes data.
    inline CRefData(tSize Size, const tData& Value);

    //! Constructor that copies a specified data.
    inline CRefData(tConstDataPtr pData, tSize Size);

    //! Constructor copies a given data.
    inline CRefData(const CRefData& Data);

    //! Constructor which makes a reference to an existing data.
    //! - Can be called in this way CRefData(Data, REFERENCE).
    inline CRefData(const CRefData& Data, EMakeRef);

    //! Destructor.
    inline ~CRefData();

    //! Assignment operator.
    inline CRefData& operator=(const CRefData& Data);

    //! Allocates data of a given size.
    //! - This method is primarily designed for data reference initialization
    //!   when the default constructor was used.
    //! - Provides no checking of the existing data and its (de,re)allocation.
    //! - Avoid of using it!
    inline void init(tSize Size);

    //! Allocates data of a given size.
    //! - In case of several references to the existing data, the reference
    //!   counter is decremented and new data are allocated.
    inline void resize(tSize Size);

    //! Allocates and initializes data.
    inline void create(tSize Size, const tData& Value);

    //! Allocates data and copies the specified data to it.
    inline void create(tConstDataPtr pData, tSize Size);

    //! Allocates data and copies the specified data.
    inline void create(const CRefData& Data);

    //! Makes a new reference to existing data.
    //! - Eventually deallocates the associated data.
    inline void create(const CRefData& Data, EMakeRef);


    //! Returns data size.
    tSize getSize() const { return m_pRef->m_Size; }

    //! Returns the number of references to the data.
    int getNumOfReferences() const { return m_pRef->m_iReferences; }

    //! Returns the subscripted element [i].
    T& operator ()(tSize i)
    {
        return m_pRef->m_pData[i];
    }
    const T& operator ()(tSize i) const
    {
        return m_pRef->m_pData[i];
    }

    //! Gets the element [i].
    T& get(tSize i)
    {
        return m_pRef->m_pData[i];
    }
    const T& get(tSize i) const
        {
            return m_pRef->m_pData[i];
        }

    //! Returns pointer to the given element.
    T *getPtr(tSize i)
    {
        return (m_pRef->m_pData + i);
    }
    const T *getPtr(tSize i) const
    {
        return (m_pRef->m_pData + i);
    }

    //! Sets the element of matrix at the position [i].
    void set(tSize i, const T& Value)
    {
        m_pRef->m_pData[i] = Value;
    }

    //! Returns pointer to the data.
    tDataPtr getPtr()
    {
        return m_pRef->m_pData;
    }
    const tDataPtr getPtr() const
    {
        return m_pRef->m_pData;
    }

protected:
    //! Structure containing information on allocated data
    //! and the number of references to this data.
    struct SRefInfo /*: public CSmallValueObject<>*/
    {
        //! Size of the allocated data.
        tSize m_Size;

        //! Pointer to the allocated data.
        tDataPtr m_pData;

        //! The number of references to the data.
        //! - Zero if there is no reference.
        int m_iReferences;

        //! Constructor that allocates a given size of data.
        //! - Allocation of the zero size data is allowed.
        SRefInfo(tSize Size) : m_Size(Size), m_iReferences(1)
        {
            MDS_ASSERT(Size >= 0);

            m_pData = memAlloc<tData>(m_Size);
        }

        //! Destructor.
        //! - Deallocates the data.
        ~SRefInfo() { memFree(m_pData); }
    };

    //! Pointer to the structure containing the referenced data.
    SRefInfo *m_pRef;
};


//==============================================================================
/*
 * Method templates
 */

// Include file containing method templates
#include "mdsRefData.hxx"


} // namespace base
} // namespace mds

#endif // MDS_REFDATA_H

