//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsObject.h                         \n
 * Section: libBase                             \n
 * Date:    2004/11/22                          \n
 *
 * $Id: mdsObject.h 371 2007-06-18 14:52:08Z spanel $
 *
 * Description:
 * - Derived objects can be registered in an object factory,
 *   thus created throught that factory.
 * - Provides intrusive reference counting.
 * - Derived classes should be handled using smart pointer CSharedPtr.
 */

#ifndef MDS_OBJECT_H
#define MDS_OBJECT_H

#include "mdsSetup.h"
#include "mdsAssert.h"

// Standard C++ Library
#include <new>
#include <cstddef>


namespace mds
{

//==============================================================================
/*
 * Several useful macros.
 */

//! Macro used to add standard method getClassName() to a class.
#define MDS_CLASS_NAME(name) \
    virtual const char *getClassName() const \
    { \
        return #name; \
    }

//! Special macro used to add standard method getClassName() to a class template.
//! Name containes information about concrete template parameters.
//! - Parameter 'Type' is a string literal or C/STL string.
#define MDS_TEMPLATE_NAME(Name, Type) \
    virtual const char *getClassName() const \
    { \
        static const std::string ssName = std::string(#Name) + '<' + std::string(Type) + '>'; \
        return ssName.c_str(); \
    }

//! Macro used to add static method getClassName() to a class.
#define MDS_STATIC_NAME(name) \
    static const char *getClassName() \
    { \
        return #name; \
    }

//! Special macro used to add static method getClassName() to a class template.
//! Name containes information about concrete template parameters.
//! - Parameter 'Type' is a string literal or C/STL string.
#define MDS_STATIC_TEMPLATE_NAME(Name, Type) \
    static const char *getClassName() const \
    { \
        static const std::string ssName = std::string(#Name) + '<' + std::string(Type) + '>'; \
        return ssName.c_str(); \
    }


//==============================================================================
/*!
 * Basic class providing intrusive reference counting model.
 * - It must be used as first, not virtual, base class.
 * - Mechanism used to determine if an object is on the heap is not functional
 *   when you use the operator new[] to allocate an array of objects! This may
 *   cause some memory leaks.
 */
class CObject
{
public:
    //! Templates that require members of the CObject class can use
    //! this enum to check the existence.
    enum { CLASS_OBJECT };

    //! Standard method getClassName().
    MDS_CLASS_NAME(CObject);

    //! Signature used to recognize objects on the heap.
    enum { HEAP_OBJECT = 0x70616568 };

public:
    //! Allocates a new object on the heap.
    static void *operator new(std::size_t Size)
    {
        // Get block of correct size
        void *pBlock = ::operator new(Size);

        // Point into the object-to-be and set its heap object flag
        CObject *pObject = static_cast<CObject *>(pBlock);
        pObject->m_iHeapObject = HEAP_OBJECT;
        return pBlock;
    }

    //! Virtual destructor.
    virtual ~CObject() { m_iHeapObject = 0; }

    //! Returns the number of references.
    int getReferencesCount() const
    {
        return m_iReferences;
    }

    //! Returns true if the object is allocated on the heap.
    bool isOnHeap() const
    {
        return (m_iHeapObject == HEAP_OBJECT);
    }

protected:
    //! The number of references to the object.
    //! - Zero if there is no reference.
    mutable int m_iReferences;

    //! Flag initialized if the object is on the heap.
    int m_iHeapObject;

protected:
    //! Protected default constructor.
    //! - If the object is created statically, initialize the refereces counter
    //!   to one, otherwise to zero.
    CObject() : m_iReferences((m_iHeapObject == HEAP_OBJECT) ? 0 : 1) {}

    //! Increase the reference counter.
    void addReference() const
    {
        ++m_iReferences;
    }

    //! Decrease the reference counter.
    //! - Returns true if the decremented number of references
    //!   is lower or equal to zero and object must be deleted.
    bool delReference() const
    {
        return ((--m_iReferences) == 0);
    }

    //! Protected copy constructor.
    CObject(const CObject&);

    //! Protected assignment operator.
    void operator=(const CObject&);
};


} // namespace mds

#endif // MDS_OBJECT_H

