//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 * 
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsMemory.h                         \n
 * Section: libBase                             \n
 * Date:    2004/12/23                          \n
 *
 * $Id: mdsMemory.h 92 2006-09-24 09:53:39Z spanel $
 *
 * Description:
 * - Memory allocation and manipulation functions.
 */

#ifndef MDS_MEMORY_H
#define MDS_MEMORY_H

#include "mdsSetup.h"
#include "mdsAssert.h"
#include "mdsTypes.h"
#include "mdsError.h"

#include <malloc.h>
#include <memory.h>


namespace mds
{

//==============================================================================
/*
 * Memory allocation functions.
 */

//! Allocates memory.
template <typename T>
inline T *memAlloc(tSize Count)
{
    T *p = (T *)malloc(Count * sizeof(T));
    if( !p )
    {
        MDS_THROW_ERROR("Function memAlloc() failed: Cannot allocate memory");
    }
    return p;
}


//! De-allocates memory.
template <typename T>
inline void memFree(T *pMem)
{
    free((void *)pMem);
}


//==============================================================================
/*
 * Basic memory manipulation functions.
 */

//! Copies memory.
template <typename T>
inline void memCopy(T *pDst, const T *pSrc, tSize Count)
{
    memcpy((void *)pDst, (void *)pSrc, Count * sizeof(T));
}


//! Copies memory.
template <typename T>
inline void memLoopCopy(T *pDst, const T *pSrc, tSize Count)
{
    T *p = pDst;
    T *pMax = pDst + Count;
    const T *pS = pSrc;
    while( p < pMax )
    {
        *(p++) = *(pS++);
    }
}


//! Initializes memory.
template <typename T>
inline void memSet(T *pMem, const T& Value, tSize Count)
{
    T *p = pMem;
    T *pMax = pMem + Count;
    while( p < pMax )
    {
        *(p++) = Value;
    }
}


//! Swaps content of two memory blocks.
template <typename T>
inline void memSwap(T *pMem, T *pMem2, tSize Count)
{
    T *p = pMem;
    T *pMax = pMem + Count;
    T *p2 = pMem2;
    while( p < pMax )
    {
        T Temp = *p;
        *(p++) = *p2;
        *(p2++) = Temp;
    }
}


//==============================================================================
/*
 * Basic memory manipulation functions.
 * - Sparse data vector.
 */

//! Copies memory.
template <typename T>
inline void memSparseCopy(T *pDst, tSize DstStride, const T *pSrc, tSize SrcStride, tSize Count)
{
    T *p = pDst;
    T *pMax = pDst + Count * DstStride;
    const T *pS = pSrc;
    while( p < pMax )
    {
        *p = *pS;
        p += DstStride;
        pS += SrcStride;
    }
}


//! Initializes memory.
template <typename T>
inline void memSparseSet(T *pMem, tSize Stride, const T& Value, tSize Count)
{
    T *p = pMem;
    T *pMax = pMem + Count * Stride;
    while( p < pMax )
    {
        *p = Value;
        p += Stride;
    }
}


//! Swaps content of two memory blocks.
template <typename T>
inline void memSparseSwap(T *pMem, tSize Stride, const T *pMem2, tSize Stride2, tSize Count)
{
    T *p = pMem;
    T *pMax = pMem + Count * Stride;
    const T *p2 = pMem2;
    while( p < pMax )
    {
        T Temp = *p;
        *p = *p2;
        *p2 = Temp;

        p += Stride;
        p2 += Stride2;
    }
}


//==============================================================================
/*
 * Extended memory manipulation functions.
 */

//! Adds 'Value' to all memory elements.
template <typename T, typename U>
inline void memAdd(T *pMem, const U& Value, tSize Count)
{
    T *p = pMem;
    T *pMax = pMem + Count;
    while( p < pMax )
    {
        *(p++) += Value;
    }
}


//! Subtracts 'Value' from memory elements.
template <typename T, typename U>
inline void memSub(T *pMem, const U& Value, tSize Count)
{
    T *p = pMem;
    T *pMax = pMem + Count;
    while( p < pMax )
    {
        *(p++) -= Value;
    }
}


//! Multiples memory elements by 'Value'.
template <typename T, typename U>
inline void memMult(T *pMem, const U& Value, tSize Count)
{
    T *p = pMem;
    T *pMax = pMem + Count;
    while( p < pMax )
    {
        *(p++) *= Value;
    }
}


//! Divides memory elements by 'Value'.
template <typename T, typename U>
inline void memDiv(T *pMem, const U& Value, tSize Count)
{
    T *p = pMem;
    T *pMax = pMem + Count;
    while( p < pMax )
    {
        *(p++) /= Value;
    }
}


//! Adds memory elements pointed to by 'pSrc' to 'pDst'.
template <typename T, typename U>
inline void memVectAdd(T *pDst, const U *pSrc, tSize Count)
{
    T *p = pDst;
    T *pMax = pDst + Count;
    const U *pS = pSrc;
    while( p < pMax )
    {
        *(p++) += *(pS++);
    }
}


//! Adds memory elements pointed to by 'pSrc' and multiplied by 'Value' to 'pDst'.
template <typename T, typename U, typename V>
inline void memVectMultAdd(T *pDst, const U *pSrc, const V& Value, tSize Count)
{
    T *p = pDst;
    T *pMax = pDst + Count;
    const U *pS = pSrc;
    while( p < pMax )
    {
        *(p++) += Value * *(pS++);
    }
}


//! Subtracts memory elements pointed to by 'pSrc' from 'pDst'.
//! - Result is stored in 'pDst'.
template <typename T, typename U>
inline void memVectSub(T *pDst, const U *pSrc, tSize Count)
{
    T *p = pDst;
    T *pMax = pDst + Count;
    const U *pS = pSrc;
    while( p < pMax )
    {
        *(p++) -= *(pS++);
    }
}


//! Multiples memory elements pointed to by 'pDst' by 'pSrc' elements.
template <typename T, typename U>
inline void memVectMult(T *pDst, const U *pSrc, tSize Count)
{
    T *p = pDst;
    T *pMax = pDst + Count;
    const U *pS = pSrc;
    while( p < pMax )
    {
        *(p++) *= *(pS++);
    }
}


//! Divides memory elements pointed to by 'pDst' by 'pSrc' elements.
template <typename T, typename U>
inline void memVectDiv(T *pDst, const U *pSrc, tSize Count)
{
    T *p = pDst;
    T *pMax = pDst + Count;
    const U *pS = pSrc;
    while( p < pMax )
    {
        *(p++) /= *(pS++);
    }
}


//==============================================================================
/*
 * Extended memory manipulation functions.
 * - Sparse vector of data.
 */

//! Adds 'Value' to all memory elements.
template <typename T, typename U>
inline void memSparseAdd(T *pMem, tSize Stride, const U& Value, tSize Count)
{
    T *p = pMem;
    T *pMax = pMem + Count * Stride;
    while( p < pMax )
    {
        *p += Value;
        p += Stride;
    }
}


//! Subtracts 'Value' from memory elements.
template <typename T, typename U>
inline void memSparseSub(T *pMem, tSize Stride, const U& Value, tSize Count)
{
    T *p = pMem;
    T *pMax = pMem + Count * Stride;
    while( p < pMax )
    {
        *p -= Value;
        p += Stride;
    }
}


//! Multiples memory elements by 'Value'.
template <typename T, typename U>
inline void memSparseMult(T *pMem, tSize Stride, const U& Value, tSize Count)
{
    T *p = pMem;
    T *pMax = pMem + Count * Stride;
    while( p < pMax )
    {
        *p *= Value;
        p += Stride;
    }
}


//! Divides memory elements by 'Value'.
template <typename T, typename U>
inline void memSparseDiv(T *pMem, tSize Stride, const U& Value, tSize Count)
{
    T *p = pMem;
    T *pMax = pMem + Count * Stride;
    while( p < pMax )
    {
        *p /= Value;
        p += Stride;
    }
}


//! Adds memory elements pointed to by 'pSrc' to 'pDst'.
template <typename T, typename U>
inline void memVectSparseAdd(T *pDst, tSize DstStride, const U *pSrc, tSize SrcStride, tSize Count)
{
    T *p = pDst;
    T *pMax = pDst + Count * DstStride;
    const U *pS = pSrc;
    while( p < pMax )
    {
        *p += *pS;
        p += DstStride;
        pS += SrcStride;
    }
}


//! Adds memory elements pointed to by 'pSrc' and multiplied by 'Value' to 'pDst'.
template <typename T, typename U, typename V>
inline void memVectSparseMultAdd(T *pDst, tSize DstStride, const U *pSrc, tSize SrcStride, const V& Value, tSize Count)
{
    T *p = pDst;
    T *pMax = pDst + Count * DstStride;
    const U *pS = pSrc;
    while( p < pMax )
    {
        *p += Value * *pS;
        p += DstStride;
        pS += SrcStride;
    }
}


//! Subtracts memory elements pointed to by 'pSrc' from 'pDst'.
//! - Result is stored in 'pDst'.
template <typename T, typename U>
inline void memVectSparseSub(T *pDst, tSize DstStride, const U *pSrc, tSize SrcStride, tSize Count)
{
    T *p = pDst;
    T *pMax = pDst + Count * DstStride;
    const U *pS = pSrc;
    while( p < pMax )
    {
        *p -= *pS;
        p += DstStride;
        pS += SrcStride;
    }
}


//! Multiples memory elements pointed to by 'pDst' by 'pSrc' elements.
template <typename T, typename U>
inline void memVectSparseMult(T *pDst, tSize DstStride, const U *pSrc, tSize SrcStride, tSize Count)
{
    T *p = pDst;
    T *pMax = pDst + Count * DstStride;
    const U *pS = pSrc;
    while( p < pMax )
    {
        *p *= *pS;
        p += DstStride;
        pS += SrcStride;
    }
}


//! Divides memory elements pointed to by 'pDst' by 'pSrc' elements.
template <typename T, typename U>
inline void memVectSparseDiv(T *pDst, tSize DstStride, const U *pSrc, tSize SrcStride, tSize Count)
{
    T *p = pDst;
    T *pMax = pDst + Count * DstStride;
    const U *pS = pSrc;
    while( p < pMax )
    {
        *p /= *pS;
        p += DstStride;
        pS += SrcStride;
    }
}


} // namespace mds

#endif // MDS_MEMORY_H

