//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2007 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsIterator.h                       \n
 * Section: libBase                             \n
 * Date:    2005/02/02                          \n
 *
 * $Id: mdsIterator.h 345 2007-06-11 13:23:09Z spanel $
 *
 * Description:
 * - Container iterator definition.
 */

#ifndef MDS_ITERATOR_H
#define MDS_ITERATOR_H

#include "mdsSetup.h"
#include "mdsAssert.h"
#include "mdsTemplates.h"


namespace mds
{
namespace base
{

//==============================================================================
/*
 * Several useful macros.
 */

//! Macro declares a class members tIterator and tConstIterator which can be
//! later used to create simple container iterator.
#define MDS_SIMPLE_ITERATOR(I, ITER) \
    enum { ITERATOR_DECLARED }; \
    typedef ITER<I> tIterator; \
    typedef ITER<const I> tConstIterator


//==============================================================================
/*!
 * Base class for all simple iterators which don't need to hold pointer
 * to container. Only pointer to item is stored.
 * - Parameter I is item type.
 */
template <typename I>
class CSimpleIteratorBase
{
public:
    //! Container type.
    typedef tNull tContainer;

    //! Type of the item.
    typedef I tItem;

public:
    //! Default constructor
    //! - Iterator is not initialized.
    //! - Use with care!
    CSimpleIteratorBase() {}

    //! Copy constructor.
    CSimpleIteratorBase(const CSimpleIteratorBase& It) : m_pItem(It.m_pItem) {}

    //! Constructor.
    //! - Initializes the container and item pointer.
    CSimpleIteratorBase(tItem *pItem) : m_pItem(pItem) {}

    //! Assignment operator.
    CSimpleIteratorBase& operator=(const CSimpleIteratorBase& It)
    {
        m_pItem = It.m_pItem;
        return *this;
    }

    //! Destructor.
    //! - Purposely not virtual!
    ~CSimpleIteratorBase() {}


    // Operator moves the iterator to the next item.
    // - Prefix notation.
    // - Derived classes usually implements this operator.
/*    CSimpleIteratorBase& operator++() const
    {
        ++m_pItem;
        return *this;
    }*/

    // Operator moves the iterator to the next item.
    // - Postfix notation.
    // - Derived classes usually implements this operator.
/*    CSimpleIteratorBase operator++(int) const
    {
        CIteratorBase Temp(*this);
        ++m_pItem;
        return Temp;
    }*/

    // Returns the number of increments between positions addressed
    // by two iterators.
    // - Derived classes usually implements this function.
/*    tSize getDistance(const CSimpleIteratorBase& End) const
    {
        return tSize(End.m_pItem - m_pItem);
    }*/


    //! Returns pointer to the current item.
    tItem *operator->() const { return m_pItem; }

    //! Returns reference to the current item.
    tItem& operator*() const { return *m_pItem; }


    //! Enables "if (!it) ..."
    bool operator!() const { return (m_pItem == NULL); }


    //! Comparision operators.
    inline friend bool operator ==(const CSimpleIteratorBase& l, tItem *r)
    {
        return (l.m_pItem == r);
    }

    inline friend bool operator ==(tItem *l, const CSimpleIteratorBase& r)
    {
        return (r.m_pItem == l);
    }

    inline friend bool operator !=(const CSimpleIteratorBase& l, tItem *r)
    {
        return (l.m_pItem != r);
    }

    inline friend bool operator !=(tItem *l, const CSimpleIteratorBase& r)
    {
        return (r.m_pItem != l);
    }

    inline friend bool operator <(const CSimpleIteratorBase& l, tItem * r)
    {
        return (l.m_pItem < r);
    }

    inline friend bool operator <(tItem* l, const CSimpleIteratorBase& r)
    {
        return (l < r.m_pItem);
    }

    inline friend bool operator <=(const CSimpleIteratorBase& l, tItem *r)
    {
        return (l.m_pItem <= r);
    }

    inline friend bool operator <=(tItem *l, const CSimpleIteratorBase& r)
    {
        return (l <= r.m_pItem);
    }

    //! Ambiquity buster that solves problem using comparision
    //! like "if( p == sp ) ..."
    template <typename U>
    inline friend bool operator ==(const CSimpleIteratorBase& l, const U *r)
    {
        return (l.m_pItem == r);
    }

    template <typename U>
    inline friend bool operator ==(const U *l, const CSimpleIteratorBase& r)
    {
        return (r.m_pItem == l);
    }

    template <typename U>
    inline friend bool operator !=(const CSimpleIteratorBase& l, const U *r)
    {
        return (l.m_pItem != r);
    }

    template <typename U>
    inline friend bool operator !=(const U *l, const CSimpleIteratorBase& r)
    {
        return (r.m_pItem != l);
    }

    template <typename U>
    inline friend bool operator <(const CSimpleIteratorBase& l, const U *r)
    {
        return (l.m_pItem < r);
    }

    template <typename U>
    inline friend bool operator <(const U *l, const CSimpleIteratorBase& r)
    {
        return (l < r.m_pItem);
    }

    template <typename U>
    inline friend bool operator <=(const CSimpleIteratorBase& l, const U *r)
    {
        return (l.m_pItem <= r);
    }

    template <typename U>
    inline friend bool operator <=(const U *l, const CSimpleIteratorBase& r)
    {
        return (l <= r.m_pItem);
    }

    //! Ambiguity buster solving the comparision of two iterators.
    template <typename U>
    bool operator ==(const CSimpleIteratorBase<U>& r) const
    {
        return (m_pItem == r.m_pItem);
    }

    template <typename U>
    bool operator!=(const CSimpleIteratorBase<U>& r) const
    {
        return (m_pItem != r.m_pItem);
    }

    template <typename U>
    bool operator<(const CSimpleIteratorBase<U>& r) const
    {
        return (m_pItem < r.m_pItem);
    }

    template <typename U>
    bool operator <=(const CSimpleIteratorBase<U>& r) const
    {
        return (m_pItem <= r.m_pItem);
    }

protected:
    //! Pointer to the container item.
    tItem *m_pItem;
};


//==============================================================================
/*
 * Several useful macros.
 */

//! Macro declares a class members tIterator and tConstIterator which can be
//! later used to create complex container iterator.
#define MDS_COMPLEX_ITERATOR(C, I, ITER) \
    enum { ITERATOR_DECLARED }; \
    typedef ITER<C,I> tIterator; \
    typedef ITER<const C, const I> tConstIterator


//==============================================================================
/*!
 * Base class for all complex container iterators which require to hold pointer
 * to container as well as pointer to concrete container item.
 * - Template parameter C is a container type.
 * - Parameter I is container item type.
 */
template <class C, typename I>
class CComplexIteratorBase
{
public:
    //! Container type.
    typedef C tContainer;

    //! Type of the container item.
    typedef I tItem;

public:
    //! Default constructor
    //! - Iterator is not initialized.
    //! - Use with care!
    CComplexIteratorBase() {}

    //! Copy constructor.
    CComplexIteratorBase(const CComplexIteratorBase& It)
        : m_pContainer(It.m_pContainer)
        , m_pItem(It.m_pItem)
    {}

    //! Constructor.
    //! - Initializes the container and item pointer.
    CComplexIteratorBase(tContainer *pContainer, tItem *pItem)
        : m_pContainer(pContainer)
        , m_pItem(pItem)
    {}

    //! Assignment operator.
    CComplexIteratorBase& operator=(const CComplexIteratorBase& It)
    {
        m_pContainer = It.m_pContainer;
        m_pItem = It.m_pItem;
        return *this;
    }

    //! Destructor.
    //! - Purposely not virtual!
    ~CComplexIteratorBase() {}


    // Operator moves the iterator to the next item.
    // - Prefix notation.
    // - Derived classes implements this operator.
/*    CComplexIteratorBase& operator++() const
    {
        ++m_pItem;
        return *this;
    }*/

    // Operator moves the iterator to the next item.
    // - Postfix notation.
    // - Derived classes implements this operator.
/*    CComplexIteratorBase operator++(int) const
    {
        CComplexIteratorBase Temp(*this);
        ++m_pItem;
        return Temp;
    }*/

    // Returns the number of increments between the positions addressed
    // by two iterators.
    // - Derived classes implements this function.
/*    tSize getDistance(const CComplexIteratorBase& End) const
    {
        // Check that both iterators addresses the same container
        if( !m_pContainer || m_pContainer != End.m_pContainer )
        {
            return 0;
        }
        
        // Count the distance
        return tSize(End.m_pItem - m_pItem);
    }*/


    //! Returns pointer to the current item.
    tItem *operator->() const { return m_pItem; }

    //! Returns reference to the current item.
    tItem& operator*() const { return *m_pItem; }


    //! Enables "if (!it) ..."
    bool operator!() const { return (m_pItem == NULL); }


    //! Comparision operators.
    inline friend bool operator ==(const CComplexIteratorBase& l, tItem *r)
    {
        return (l.m_pItem == r);
    }

    inline friend bool operator ==(tItem *l, const CComplexIteratorBase& r)
    {
        return (r.m_pItem == l);
    }

    inline friend bool operator !=(const CComplexIteratorBase& l, tItem *r)
    {
        return (l.m_pItem != r);
    }

    inline friend bool operator !=(tItem *l, const CComplexIteratorBase& r)
    {
        return (r.m_pItem != l);
    }

    inline friend bool operator <(const CComplexIteratorBase& l, tItem * r)
    {
        return (l.m_pItem < r);
    }

    inline friend bool operator <(tItem* l, const CComplexIteratorBase& r)
    {
        return (l < r.m_pItem);
    }

    inline friend bool operator <=(const CComplexIteratorBase& l, tItem *r)
    {
        return (l.m_pItem <= r);
    }

    inline friend bool operator <=(tItem *l, const CComplexIteratorBase& r)
    {
        return (l <= r.m_pItem);
    }

    //! Ambiquity buster that solves problem using comparision
    //! like "if( p == sp ) ..."
    template <typename U>
    inline friend bool operator ==(const CComplexIteratorBase& l, const U *r)
    {
        return (l.m_pItem == r);
    }

    template <typename U>
    inline friend bool operator ==(const U *l, const CComplexIteratorBase& r)
    {
        return (r.m_pItem == l);
    }

    template <typename U>
    inline friend bool operator !=(const CComplexIteratorBase& l, const U *r)
    {
        return (l.m_pItem != r);
    }

    template <typename U>
    inline friend bool operator !=(const U *l, const CComplexIteratorBase& r)
    {
        return (r.m_pItem != l);
    }

    template <typename U>
    inline friend bool operator <(const CComplexIteratorBase& l, const U *r)
    {
        return (l.m_pItem < r);
    }

    template <typename U>
    inline friend bool operator <(const U *l, const CComplexIteratorBase& r)
    {
        return (l < r.m_pItem);
    }

    template <typename U>
    inline friend bool operator <=(const CComplexIteratorBase& l, const U *r)
    {
        return (l.m_pItem <= r);
    }

    template <typename U>
    inline friend bool operator <=(const U *l, const CComplexIteratorBase& r)
    {
        return (l <= r.m_pItem);
    }

    //! Ambiguity buster solving the comparision of two iterators.
    template <class U, typename V>
    bool operator ==(const CComplexIteratorBase<U,V>& r) const
    {
        return (m_pItem == r.m_pItem);
    }

    template <class U, typename V>
    bool operator!=(const CComplexIteratorBase<U,V>& r) const
    {
        return (m_pItem != r.m_pItem);
    }

    template <class U, typename V>
    bool operator<(const CComplexIteratorBase<U,V>& r) const
    {
        return (m_pItem < r.m_pItem);
    }

    template <class U, typename V>
    bool operator <=(const CComplexIteratorBase<U,V>& r) const
    {
        return (m_pItem <= r.m_pItem);
    }

protected:
    //! Pointer to the container.
    tContainer *m_pContainer;

    //! Pointer to the container item.
    tItem *m_pItem;
};


} // namespace base
} // namespace mds

#endif // MDS_ITERATOR_H

