//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsFactory.h                        \n
 * Section: libBase                             \n
 * Date:    2005/04/03                          \n
 *
 * $Id: mdsFactory.h 64 2006-08-11 08:45:24Z spanel $
 * 
 * This code is based on The Loki Library which accompanies
 * the famous book:
 * Alexandrescu, Andrei. "Modern C++ Design: Generic Programming and Design
 *     Patterns Applied". Copyright (c) 2001. Addison-Wesley.
 *
 * The Loki Library
 * Copyright (c) 2001 by Andrei Alexandrescu
 * Permission to use, copy, modify, distribute and sell this software for any
 *     purpose is hereby granted without fee, provided that the above copyright
 *     notice appear in all copies and that both that copyright notice and this
 *     permission notice appear in supporting documentation.
 *
 * Description:
 * - Template definition of an object factory.
 */

#ifndef MDS_FACTORY_H
#define MDS_FACTORY_H

#include "mdsSetup.h"
#include "mdsAssert.h"
#include "mdsError.h"
#include "mdsLock.h"
#include "mdsSingleton.h"

// STL
#include <map>


namespace mds
{
namespace base
{

//=============================================================================
/*
 * Several usefull macros.
 */

//! Returns instance of a given object factory
#define MDS_FACTORY(T)  mds::base::CSingletonHolder<mds::base::CFactory<T::tObject, T::tIdentifier> >::getInstance()


//=============================================================================
/*!
 * Implements a generic object factory.
 * - Parameter T is a base object type
 * - Template parameter I is the object identifier type
 * - Parameter C is the object creation function. Default creation function
 *   has no parameter and returns pointer to a newly created object of type T.
 */
template <class T, typename I>
class CFactory : public CLibrarySingleton, public CLibraryLockableClass<CFactory<T,I> >
{
public:
    //! Produced object type
    typedef T tObject;

    //! Object identifier type
    typedef I tIdentifier;

    //! Object creation function
    typedef tObject *(*tCreationFunction)();

    //! Lock type
    typedef typename CLibraryLockableClass<CFactory<T,I> >::CLock tLock;

public:
    //! Destructor.
    ~CFactory() {}

    //! Registers an object creation function.
    bool registerObject(const tIdentifier& Id, tCreationFunction Function)
    {
        MDS_ASSERT(Function);

        tLock Lock(*this);
        return (m_Functions.insert(typename tFunctions::value_type(Id, Function)).second);
    }

    //! Unregisters the object creation function.
    bool unregisterObject(const tIdentifier& Id)
    {
        tLock Lock(*this);
        return (m_Functions.erase(Id) == 1);
    }

    //! Creates a new object having the identifier 'Id'.
    tObject *create(const tIdentifier& Id)
    {
        tLock Lock(*this);
        typename tFunctions::const_iterator i = m_Functions.find(Id);
        if( i != m_Functions.end() )
        {
            return (i->second)();
        }
        MDS_THROW_ERROR("Method CFactory::create() failed");
    }

private:
    //! Map of creation functions
    typedef std::map<tIdentifier, tCreationFunction> tFunctions;

    //! Map of all registered creation functions
    tFunctions m_Functions;

private:
    //! Private constructor
    CFactory() {}

    //! Allow factory instantiation using singleton holder
    MDS_PRIVATE_SINGLETON(CFactory);
};


} // namespace base
} // namespace mds

#endif // MDS_FACTORY_H

