//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsException.h                      \n
 * Section: libBase                             \n
 * Date:    2003/10/23                          \n
 *
 * $Id: mdsException.h 116 2006-10-20 08:51:07Z spanel $
 *
 * Description:
 * - Exception class.
 */

#ifndef MDS_EXCEPTION_H
#define MDS_EXCEPTION_H

#include "mdsSetup.h"

// STL
#include <string>
#include <iostream>
#include <stdexcept>


namespace mds
{

//==============================================================================
/*!
 * Class encapsulating an exception origin and description.
 */
class CException : public std::exception
{
public:
    //! Exception constructor.
    CException(const std::string& sReason,
               const std::string& sFilename,
               int iLine
               )
        : m_sReason(sReason)
        , m_sFilename(sFilename)
        , m_iLine(iLine)
    {}

    //! Exception copy constructor.
    CException(const CException& e)
        : m_sReason(e.m_sReason)
        , m_sFilename(e.m_sFilename)
        , m_iLine(e.m_iLine)
    {}

    //! Virtual destructor.
    virtual ~CException() throw() {}


    //! Returns description of exception reason.
    virtual const char *what( ) const throw();

    //! Returns description of exception reason.
    const std::string& getReason() const;

    //! Writes exception descritpion into the output stream
    //! in human readable form.
    virtual void print(std::ostream& Stream) const;


    //! Writes exception description into the output stream.
    friend std::ostream& operator <<(std::ostream& Stream, const CException& Exception);

protected:
    //! Exception reason.
    std::string m_sReason;

    //! File in which exception has risen.
    std::string m_sFilename;

    //! Line on which exception has risen.
    int m_iLine;
};


//==============================================================================
/*
 * Several macro definitions.
 */

//! Macro that creates an exception with name as an description.
//! - File and line parameters acquired from compiler.
#define MDS_EXCEPTION(Name)         mds::CException(Name, __FILE__, __LINE__)

//! Creates and throw an exception with name as an description.
//! - File and line parameters acquired from compiler.
#define MDS_THROW_EXCEPTION(Name)   throw mds::CException(Name, __FILE__, __LINE__)

//! Catch block for handling the CException exception.
//! - Writes info about it to the global log and cerr.
//! - Terminates the program.
#define MDS_CATCH_EXCEPTION(ModuleName) \
    catch( CException& Exception ) \
    { \
        MDS_LOG(Exception); \
        MDS_CERR('<' << ModuleName << "> Exception: see log for details, termintating ..." << std::endl); \
        abort(); \
    }

//! Catch block for handling STL and unknown exceptions.
//! - Writes info about it to the global log and cerr.
//! - Terminates the program.
#define MDS_CATCH_UNKNOWN_EXCEPTION(ModuleName) \
    catch( std::exception& Exception ) \
    { \
        MDS_LOG('<' << ModuleName << "> Std exception: " << Exception.what() << std::endl); \
        MDS_CERR('<' << ModuleName << "> Std exception: see log for details, termintating ..." << std::endl); \
        abort(); \
    } \
    catch( ... ) \
    { \
        MDS_CERR('<' << ModuleName << "> Unknown exception: termintating ..." << std::endl); \
        abort(); \
    }


} // namespace mds

#endif // MDS_EXCEPTION_H

