//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsError.h                          \n
 * Section: libBase                             \n
 * Date:    2003/10/27                          \n
 *
 * $Id: mdsError.h 116 2006-10-20 08:51:07Z spanel $
 *
 * Description:
 * - Error class.
 */

#ifndef MDS_ERROR_H
#define MDS_ERROR_H

#include "mdsSetup.h"
#include "mdsException.h"


namespace mds
{

//==============================================================================
/*!
 * Class encapsulating an error origin and description.
 */
class CError : public CException
{
public:
    //! Error constructor.
    CError(const std::string& sReason,
           const std::string& sFileName,
           int iLine
           )
        : CException(sReason, sFileName, iLine)
    {}

    //! Copy constructor.
    CError(const CError& e) : CException(e) {}

    //! Virtual destructor.
    virtual ~CError() throw() {}


    //! Writes error description into the output stream
    //! in human readable form.
    virtual void print(std::ostream& Stream) const;

    //! Writes error description into the output stream.
    friend std::ostream& operator <<(std::ostream& os, const CError& Error);
};


//==============================================================================
/*
 * Several useful macro definitions.
 */

//! Macro that creates an error object with 'Name' as an description.
//! - File and line parameters acquired from a compiler.
#define MDS_ERROR(Name)         mds::CError(Name, __FILE__, __LINE__)

//! Creates and throws an error with 'Name' as an description.
//! - File and line parameters acquired from a compiler.
#define MDS_THROW_ERROR(Name)   throw mds::CError(Name, __FILE__, __LINE__)

//! Catch block for handling the CError exception.
//! - Writes info about it to the global log and cerr.
//! - Terminates the program.
#define MDS_CATCH_ERROR(ModuleName) \
    catch( CError& Error ) \
    { \
        MDS_LOG(Error); \
        MDS_CERR('<' << ModuleName << "> Error: see log for details, termintating ..." << std::endl); \
        abort(); \
    }


} // namespace mds

#endif // MDS_ERROR_H

