//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsData.hxx                         \n
 * Section: libBase                             \n
 * Date:    2004/11/23                          \n
 *
 * $Id: mdsData.hxx 64 2006-08-11 08:45:24Z spanel $
 * 
 * Description:
 * - Template providing data allocation functions.
 * - Allocation using malloc() and free().
 */

//==============================================================================
/*
 * Methods templates.
 */

// Default constructor
template <typename T>
inline CData<T>::CData()
{
}


// Constructor that allocates a given size of data
template <typename T>
inline CData<T>::CData(tSize Size)
    : m_Size(Size)
{
    MDS_ASSERT(m_Size >= 0);

    m_pData = memAlloc<tData>(m_Size);
}


// Allocates and initializes data
template <typename T>
inline CData<T>::CData(tSize Size, const tData& Value)
    : m_Size(Size)
{
    MDS_ASSERT(m_Size >= 0);

    m_pData = memAlloc<tData>(m_Size);

    memSet(m_pData, Value, m_Size);
}


// Constructor that copies a specified data
template <typename T>
inline CData<T>::CData(tConstDataPtr pData, tSize Size)
    : m_Size(Size)
{
    MDS_ASSERT(m_Size >= 0 && pData);

    m_pData = memAlloc<tData>(m_Size);

    memCopy(m_pData, pData, m_Size);
}


// Constructor copies a given data
template <typename T>
inline CData<T>::CData(const CData<T>& Data)
    : m_Size(Data.m_Size)
{
    m_pData = memAlloc<tData>(m_Size);

    memCopy(m_pData, Data.m_pData, m_Size);
}


// Virtual destructor
template <typename T>
inline CData<T>::~CData()
{
    memFree(m_pData);
}


// Assignment operator
template <typename T>
inline CData<T>& CData<T>::operator=(const CData<T>& Data)
{
    if( this != &Data )
    {
        create(Data);
    }
    return *this;
}


// Allocates memory for data
template <typename T>
inline void CData<T>::init(tSize Size)
{
    MDS_ASSERT(Size >= 0);

    m_Size = Size;
    m_pData = memAlloc<tData>(m_Size);
}


// Allocates memory for data
template <typename T>
inline void CData<T>::resize(tSize Size)
{
    MDS_ASSERT(Size >= 0);

    if( Size > m_Size )
    {
        memFree(m_pData);
        m_Size = Size;
        m_pData = memAlloc<tData>(m_Size);
    }
    else
    {
        m_Size = Size;
    }
}


// Allocates and initializes data
template <typename T>
inline void CData<T>::create(tSize Size, const tData& Value)
{
    resize(Size);

    memSet(m_pData, Value, m_Size);
}


// Allocates new memory and copies the specified data to it
template <typename T>
inline void CData<T>::create(tConstDataPtr pData, tSize Size)
{
    resize(Size);

    memCopy(m_pData, pData, m_Size);
}


// Allocates new memory and copies the specified data
template <typename T>
inline void CData<T>::create(const CData<T>& Data)
{
    resize(Data.m_Size);

    memCopy(m_pData, Data.m_pData, m_Size);
}

