//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsData.h                           \n
 * Section: libBase                             \n
 * Date:    2004/11/23                          \n
 *
 * $Id: mdsData.h 175 2006-12-13 08:28:34Z spanel $
 * 
 * Description:
 * - Template providing data allocation functions.
 * - Allocation using malloc() and free().
 */

#ifndef MDS_DATA_H
#define MDS_DATA_H

#include "mdsSetup.h"
#include "mdsAssert.h"
#include "mdsTypes.h"
#include "mdsMemory.h"


namespace mds
{
namespace base
{

//==============================================================================
/*!
 * Template providing data allocation functions.
 */
template <typename T>
class CData
{
public:
    //! Data type
    typedef T tData;

    //! Pointer to the data
    typedef T *tDataPtr;

    //! Pointer to the constant data
    typedef const T *tConstDataPtr;

public:
    //! Default constructor.
    //! - It does no allocation of the data.
    //! - Be sure that the method init(...) is used somewhere.
    //! - Avoid of using it!
    inline CData();

    //! Constructor that allocates a given size of data.
    //! - Data of the zero size can be also allocated.
    inline CData(tSize Size);

    //! Constructor allocates and initializes data.
    inline CData(tSize Size, const tData& Value);

    //! Constructor that copies a specified data.
    inline CData(tConstDataPtr pData, tSize Size);

    //! Constructor copies a given data.
    inline CData(const CData& Data);

    //! Destructor.
    //! - Deallocates the data.
    inline ~CData();

    //! Assignment operator
    inline CData& operator=(const CData& Data);

    //! Allocates data of a given size.
    //! - This method is primarily designed for data allocation
    //!   when the default constructor was used.
    //! - Provides no checking of the existing data and its (de,re)allocation.
    //! - Avoid of using it!
    inline void init(tSize Size);

    //! Allocates memory for data of a given size.
    inline void resize(tSize Size);

    //! Allocates and initializes data.
    inline void create(tSize Size, const tData& Value);

    //! Allocates memory and copies the specified data to it.
    inline void create(tConstDataPtr pData, tSize Size);

    //! Allocates memory and copies the specified data.
    inline void create(const CData& Data);


    //! Returns data size.
    tSize getSize() const { return m_Size; }

    //! Returns the subscripted element [i].
    T& operator ()(tSize i)
    {
        return m_pData[i];
    }
    const T& operator ()(tSize i) const
    {
        return m_pData[i];
    }

    //! Gets the element [i].
    T& get(tSize i)
    {
        return m_pData[i];
    }
    const T& get(tSize i) const
    {
        return m_pData[i];
    }

    //! Returns pointer to the given element.
    T *getPtr(tSize i)
    {
        return (m_pData + i);
    }
    const T *getPtr(tSize i) const
    {
        return (m_pData + i);
    }

    //! Sets the element of matrix at the position [i].
    void set(tSize i, const T& Value)
    {
        m_pData[i] = Value;
    }

    //! Returns pointer to the data.
    tDataPtr getPtr()
    {
        return m_pData;
    }
    const tDataPtr getPtr() const
    {
        return m_pData;
    }

protected:
    //! Size of the allocated data.
    tSize m_Size;

    //! Pointer to the allocated data.
    tDataPtr m_pData;
};


//==============================================================================
/*
 * Method templates
 */

// Include file containing method templates
#include "mdsData.hxx"


} // namespace base
} // namespace mds

#endif // MDS_DATA_H

