//==============================================================================
/*! \file
 * Medical Data Segmentation Toolkit (MDSTk)    \n
 * Copyright (c) 2003-2005 by Michal Spanel     \n
 *
 * Author:  Michal Spanel, spanel@fit.vutbr.cz  \n
 * File:    mdsBasePtr.h                        \n
 * Section: libBase                             \n
 * Date:    2004/10/17                          \n
 *
 * $Id: mdsBasePtr.h 375 2007-06-19 14:42:03Z spanel $
 *
 * Description:
 * - Base smart pointer which makes no copy of a handled object.
 * - Pointer ownning the object has the bDestroy flag set to true.
 */

#ifndef MDS_BASEPTR_H
#define MDS_BASEPTR_H

#include "mdsSetup.h"
#include "mdsAssert.h"


namespace mds
{
namespace base
{

//==============================================================================
/*
 * Global macros.
 */

//! Macro declares a class member tSmartPtr which can be later used to create
//! smart pointer to class T. It also allows class instantiation by smart
//! pointer only. So it should be located in the private part of the class
//! as well as default constructor, copy constructor and all others
//! constructors.
#define MDS_BASEPTR(T) \
    typedef mds::base::CBasePtr<T> tSmartPtr; \
    friend class mds::base::CBasePtr<T>


//==============================================================================
/*!
 * Base copying smart pointer. Once the pointer is initialized, it takes full
 * control over the pointer including deletion.
 */
template <typename T>
class CBasePtr
{
public:
    //! Stored pointer type.
    typedef T *tPointer;

    //! Pointee type.
    typedef T tPointee;

public:
    //! Default constructor.
    //! - Internal pointer is initialized to the NULL.
    //! - Use with care!
    CBasePtr() : m_pInstance(0), m_bDestroy(false) {}

/*    //! Constructor.
    //! - Uses pointee type constructor.
    //! - Obsolete!
    template <typename T1>
    CBasePtr(const T1& p1) : m_pInstance(new T(p1)), m_bDestroy(true)
    {
        MDS_ASSERT(m_pInstance);
    }*/

    //! Constructor from a raw pointer.
    CBasePtr(T *p) : m_pInstance(p), m_bDestroy(true) {}

    //! Copy constructor.
    //! - Transfers ownership from one pointer to another.
    CBasePtr(CBasePtr& p)
        : m_pInstance(p.m_pInstance)
        , m_bDestroy(p.m_bDestroy)
    {
        p.m_bDestroy = false;
    }

/*    //! Constructor.
    //! - Uses pointee type constructor.
    //! - Obsolete!
    template <typename T1, typename T2>
    CBasePtr(const T1& p1, const T2& p2)
        : m_pInstance(new T(p1, p2))
        , m_bDestroy(true)
    {
        MDS_ASSERT(m_pInstance);
    }

    //! Constructor.
    //! - Uses pointee type constructor.
    //! - Obsolete!
    template <typename T1, typename T2, typename T3>
    CBasePtr(const T1& p1, const T2& p2, const T3& p3)
        : m_pInstance(new T(p1, p2, p3))
        , m_bDestroy(true)
    {
        MDS_ASSERT(m_pInstance);
    }

    //! Constructor.
    //! - Uses pointee type constructor.
    //! - Obsolete!
    template <typename T1, typename T2, typename T3, typename T4>
    CBasePtr(const T1& p1, const T2& p2, const T3& p3, const T4& p4)
        : m_pInstance(new T(p1, p2, p3, p4))
        , m_bDestroy(true)
    {
        MDS_ASSERT(m_pInstance);
    }

    //! Constructor
    //! - Uses pointee type constructor.
    //! - Obsolete!
    template <typename T1, typename T2, typename T3, typename T4, typename T5>
    CBasePtr(const T1& p1, const T2& p2, const T3& p3, const T4& p4, const T5& p5)
        : m_pInstance(new T(p1, p2, p3, p4, p5))
        , m_bDestroy(true)
    {
        MDS_ASSERT(m_pInstance);
    }

    //! Constructor.
    //! - Uses pointee type constructor.
    //! - Obsolete!
    template <typename T1, typename T2, typename T3, typename T4, typename T5, typename T6>
    CBasePtr(const T1& p1, const T2& p2, const T3& p3, const T4& p4, const T5& p5, const T6& p6)
        : m_pInstance(new T(p1, p2, p3, p4, p5, p6))
        , m_bDestroy(true)
    {
        MDS_ASSERT(m_pInstance);
    }*/


    //! Assignment operator.
    //! - Transfers ownership from one pointer to another.
    CBasePtr& operator=(CBasePtr& p)
    {
        if( this != &p )
        {
            if( m_bDestroy )
            {
                delete m_pInstance;
            }
            m_pInstance = p.m_pInstance;
            m_bDestroy = p.m_bDestroy;
            p.m_bDestroy = false;
        }
        return *this;
    }

    //! Assignment operator
    //! - Makes a reference to the object
    CBasePtr& operator=(T *p)
    {
        if( m_pInstance != p )
        {
            if( m_bDestroy )
            {
                delete m_pInstance;
            }
            m_pInstance = p;
            m_bDestroy = true;
        }
        return *this;
    }

    //! Destructor
    ~CBasePtr()
    {
        if( m_bDestroy )
        {
            delete m_pInstance;
        }
    }


    //! Conversion to the raw pointer
    operator T*() { return m_pInstance; }

    //! Added conversion to the void pointer to avoid raw pointer deletion
    operator void*() { return m_pInstance; }


    //! Returns pointer to the encapsulated object
    T *operator->() const { return m_pInstance; }

    //! Returns reference to the stored object
    T& operator*() const { return *m_pInstance; }


    //! Enables "if (!sp) ..."
    bool operator!() const { return (m_pInstance == 0); }


    //! Comparision operators
    inline friend bool operator==(const CBasePtr& l, const T* r)
    {
        return (l.m_pInstance == r);
    }

    inline friend bool operator==(const T* l, const CBasePtr& r)
    {
        return (r.m_pInstance == l);
    }

    inline friend bool operator!=(const CBasePtr& l, const T* r)
    {
        return (l.m_pInstance != r);
    }

    inline friend bool operator!=(const T* l, const CBasePtr& r)
    {
        return (r.m_pInstance != l);
    }

    inline friend bool operator<(const CBasePtr& l, const T* r)
    {
        return (l.m_pInstance < r);
    }

    inline friend bool operator<(const T* l, const CBasePtr& r)
    {
        return (r.m_pInstance > l);
    }

    //! Ambiquity buster that solves problem using comparision
    //! like "if( p == sp ) ..."
    template <typename U>
    inline friend bool operator==(const CBasePtr& l, const U* r)
    {
        return (l.m_pInstance == r);
    }

    template <typename U>
    inline friend bool operator==(const U* l, const CBasePtr& r)
    {
        return (r.m_pInstance == l);
    }

    template <typename U>
    inline friend bool operator!=(const CBasePtr& l, const U* r)
    {
        return (l.m_pInstance != r);
    }

    template <typename U>
    inline friend bool operator!=(const U* l, const CBasePtr& r)
    {
        return (r.m_pInstance != l);
    }

    template <typename U>
    inline friend bool operator<(const CBasePtr& l, const U* r)
    {
        return (l.m_pInstance < r);
    }

    template <typename U>
    inline friend bool operator<(const U* l, const CBasePtr& r)
    {
        return (r.m_pInstance > l);
    }

    //! Ambiguity buster solving the comparision of two smart pointers
    template <typename U>
    bool operator==(const CBasePtr<U>& r) const
    {
        return (m_pInstance == r.m_pInstance);
    }

    template <typename U>
    bool operator!=(const CBasePtr<U>& r) const
    {
        return (m_pInstance != r.m_pInstance);
    }

    template <typename U>
    bool operator<(const CBasePtr<U>& r) const
    {
        return (m_pInstance < r.m_pInstance);
    }


    //! Returns pointer to the encapsulated object.
    T *get() { return m_pInstance; }

    //! Returns pointer to the encapsulated object
    //! and clears the m_bDestroy flag.
    //! - Use with care!
    T *release()
    {
        m_bDestroy = false;
        return m_pInstance;
    }

    //! Sets the m_bDestroy flag.
    //! - Use with care!
    void setDestroyFlag() { m_bDestroy = true; }

private:
    //! Pointer to the encapsulated type
    tPointer m_pInstance;

    //! If set to false, the object won't be deleted
    bool m_bDestroy;
};


} // namespace base
} // namespace mds

#endif // MDS_BASEPTR_H

