import sys
from datetime import datetime
from src.Creator.ConcreteCreator import ConcreteCreator

class CommonParser:
    """
    Class for methods that are useful in all types of parsing
    """
    def __init__(self,metafile, one_file=False, filename="", by_sittings_or_more = False):
        self.one_file = one_file
        self.by_sittings_or_more = by_sittings_or_more
        self.filename = filename
        self.data = []
        self.output_data = {}
        self.metafile_data = metafile
        self.data_of_one_part = []
        self.user_fnc_file = self.metafile_data['config'].get('_user_function_file')
        self.deputy_ids_ok = True

    def aggregate_values(self,input_list:list, common_key:str,other_values_key:str,new_name_of_common_key=None):
        """
        Aggregates values based on a common key.

        Parameters:
        input_list (list): The list of dictionaries.
        common_key (str): The common key to aggregate values based on.
        other_values_key (str): The name of a key that values will be asigned to.
        new_name_of_common_key: new name of common_key if different naming had to be used
        """
        output_list = []
        #if new common key name was not given use the same name
        if new_name_of_common_key is None:
            new_name_of_common_key = common_key

        aggregated_values = {}

        for item in input_list:
            # Remove common key from the dictionary and change the structure
            key_value = item.pop(common_key)
            if key_value not in aggregated_values:
                aggregated_values[key_value] = {other_values_key: []}

            aggregated_values[key_value][other_values_key].append(item)

        # Reformat aggregated values
        for key, value in aggregated_values.items():
            temp_dict = {new_name_of_common_key: key}
            temp_dict.update(value)
            output_list.append(temp_dict)

        return output_list

    def combine_list_of_dicts(self, list1, list2) -> list:
        """
        Method for combining two list of dictionaries into one list of dictionaries

        Parameters:
        list1(list): First list of dictionaries
        list2(list): Second list of dictionaries
        """
        if list1 is None:
            return list2
        if list2 is None:
            return list1
        
        if isinstance(list1,dict):
            list1 = [list1]

        if isinstance(list2,dict):
            list2 = [list2]


        if len(list1) != len(list2):
            print(f'..WARNING {list1} and {list2} do not match in length!"', file=sys.stderr)
            print(f'list1_len:{len(list1)} and list2_len:{len(list2)}"', file=sys.stderr)

        # Return statement moved outside the finally block
        combined_list = []
        for dict1, dict2 in zip(list1, list2):
            combined_dict = {**dict1, **dict2}
            combined_list.append(combined_dict)
        return combined_list
    

    def combine_leaves_of_lists(self,list1, list2) ->list:
        """
        Method that recursively searches through the list and when finds dict type combine it with the second list

        Parameters:
        list1: first list to be combined
        list2: second list to be combined
        """
        if list1 is None:
            return list2
        if list2 is None:
            return list1
        
        #is not list both are dicts already
        if isinstance(list1,dict) and isinstance(list2,dict):
            list1.update(list2)
            return list1

        combined_list = []
        for item1, item2 in zip(list1, list2):
            if isinstance(item1, list) and isinstance(item2, list):
                combined_list.append(self.combine_leaves_of_lists(item1, item2))
            elif isinstance(item1, dict) or isinstance(item2, dict):
                combined_item = {}
                if isinstance(item1, dict):
                    combined_item.update(item1)
                if isinstance(item2, dict):
                    combined_item.update(item2)
                combined_list.append(combined_item)
            else:
                combined_list.append(item1)
        return combined_list
    

    def assign_ids_to_deputies_votes(self):
        """
        Method for assigning ids to deputies votes to
        """
        #sort current output
        self.output_data['zastupitelstva'][0]['zasedani'].sort(key=lambda sit_dict: datetime.strptime(sit_dict["datum"], '%Y-%m-%d'))

        for council in self.output_data['zastupitelstva']:
            if council.get('zasedani') is None:
                continue
            for sitting in council['zasedani']:
                last_sitting_date = sitting['datum']
                for voting in sitting['hlasovani']:
                    if voting.get('zastupiteleHlasy') is None:
                        continue
                    for vote in voting['zastupiteleHlasy']:
                        if self.metafile_data['zastupitelstva'][0]['zasedani']['hlasovani']['zastupiteleHlasy'].get('idZastupitel') is not None:
                            if self.metafile_data['zastupitele'].get('politickeSubjekty') is None:
                                vote['idZastupitel'] = vote['idZastupitel'].rsplit('#', 1)
                                deputy_party = vote['idZastupitel'][1]
                                vote['idZastupitel'] = vote['idZastupitel'][0].replace('#','')
                                self.update_deputy_party(vote['idZastupitel'],deputy_party,last_sitting_date)
                                vote['idZastupitel'] = self.get_deputy_id_by_fullname_party(vote['idZastupitel'], deputy_party,1)
                            else:
                                vote['idZastupitel'] = self.get_deputy_id_by_fullname_party(vote['idZastupitel'], "",0)
                        else:
                            self.give_ids(voting['zastupiteleHlasy'],'idZastupitel')
        


    


    def get_deputy_id_by_fullname_party(self,fullname,party,party_bool=bool)->int:
        """
        Method for getting id of deputy by its fullname
        """
        for deputy in self.output_data['zastupitele']:
            
            if fullname == str(deputy["jmeno"] + deputy["prijmeni"]) and party_bool == 0:
                return deputy["id"]
            elif fullname == str(deputy["jmeno"] + deputy["prijmeni"]) and deputy['politickeSubjekty'][-1]['idPolitickySubjekt'] == self.get_party_id_by_shortname(party) and party_bool == 1:
                return deputy["id"]
            
    def update_deputy_party(self,fullname:str,party:str,date):
        """
        Method for updating party for each deputy

        Parameters:
        fullname(str):fullame of deputy (firstname,lastname)
        party(str):party of deputy
        date:date of last vote
        """
        for deputy in self.output_data['zastupitele']:
            if fullname == str(deputy["jmeno"] + deputy["prijmeni"]):
                #deputy does not have assigned party
                if deputy.get('politickeSubjekty') is None:
                    deputy['politickeSubjekty'] = [{'od':date,'do':date,'idPolitickySubjekt':self.get_party_id_by_shortname(party)}]
                    return
                #same party, update date
                elif deputy['politickeSubjekty'][-1]['idPolitickySubjekt'] == self.get_party_id_by_shortname(party):
                    deputy['politickeSubjekty'][-1]['do'] = date
                    return
                #different party
                else:
                    #names can be duplicate, if it was already found, try another person
                    if date == deputy['politickeSubjekty'][-1]['do']:
                        continue
                    deputy['politickeSubjekty'].append({'od':date,'do':date,'idPolitickySubjekt':self.get_party_id_by_shortname(party)})

    def get_party_id_by_shortname(self,name) -> int:
        """Method for getting id of party by its shortened name"""
        for party in self.output_data['politickeSubjekty']:
            if name == str(party["zkrNazev"]):
                return party['id']


    
    def give_ids(self,list:list,id_name:str):
        """
        Method for giving ids to all items in a list

        Parameters:
        list:list that has values that need id
        id_name: name of the id (key ids will be assigned to)
        """
        for id,item in enumerate(list):
            item[id_name] = id + 1


    def delete_one_of(self):
        """
        Method deletes values that are written in the model as OneOf
        """
        self.delete_one_of_votes_summary()

    def delete_deputies_with_no_party(self):
        deputy_with_party_list = []
        for deputy in self.output_data['zastupitele']:
            if deputy.get('politickeSubjekty') is None:
                continue
            deputy_with_party_list.append(deputy)
        self.output_data['zastupitele'] = deputy_with_party_list
        if len(deputy_with_party_list) == len(self.output_data['zastupitele']):
            self.deputy_ids_ok = True
            return
        self.deputy_ids_ok = False
        return
    

    def update_deputies_id(self):
        if self.deputy_ids_ok == True:
            return
        
        

        deputy_tuple_list = []
        for index, deputy in enumerate(self.output_data['zastupitele']):
            new_id = index+1
            deputy_tuple_list.append((deputy['id'], new_id))
            deputy['id'] = new_id

        for council in self.output_data['zastupitelstva']:
            for lidr in council['lidr']:
                for deputy in deputy_tuple_list:
                    if lidr['idZastupitel'] == deputy[0]:
                        lidr['idZastupitel'] = deputy[1]
                        break
            print(council,file=sys.stderr)
            for sitting in council['zasedani']:
                for voting in sitting['hlasovani']:
                    for vote in voting['zastupiteleHlasy']:
                        for deputy in deputy_tuple_list:
                            if vote['idZastupitel'] == deputy[0]:
                                vote['idZastupitel'] = deputy[1]
                                break

        return

    def delete_one_of_votes_summary(self):
        """
        Method for deleting summary if votes and summary are both used
        """
        for council in self.output_data['zastupitelstva']:
            if council.get('zasedani') is None:
                continue
            for sitting in council['zasedani']:
                for voting in sitting['hlasovani']:
                    if voting.get('sumarizace') is not None and voting.get('zastupiteleHlasy') is not None:
                        del voting['sumarizace']

    def convert_string_to_list(self, variable):
        """Method that wraps string variable into list"""
        if isinstance(variable, str) or isinstance(variable, bool) or isinstance(variable,int):
            return [variable]
        else:
            return variable

    def apply_type_rule_on_list(self,list,value_type=None,value_params=None)->list:
        """Method that applies type on each element in a list"""
        creator = ConcreteCreator()
        #if what came in was a string we have to return str as well
        is_str = 0
        updated_list = []
        if value_type is None:
            return list
        if isinstance(list,str):
            is_str = 1
            list = [list]
        if list is None:
            list = [list]
        if value_type == "FunctionStringType" and value_params.get("_apply_on_list") == True:
            updated_list.append(creator.get_value_by_type(list,value_type,value_params,self.user_fnc_file,self.filename))
        else:
            for element in list:
                updated_list.append(creator.get_value_by_type(element,value_type,value_params,self.user_fnc_file,self.filename))
        if is_str == 1:
            #initial value was string, return as a string
            return updated_list[0]
        return updated_list
    
    def apply_list_type_rule_on_list(self,lst, key,list_type=None)->list:
        """
        Method that applies list type on list
        
        Parameters: 
        lst: input_list
        key: key of the values are saved under
        list_type: type for list manipulation

        Returns:
        list: updated list
        """
        creator = ConcreteCreator()
        updated_list = []
        if list_type is None:
            return lst
        if isinstance(lst,dict):
            lst = [lst]
        updated_list = creator.get_list_value_by_type(lst,list_type,key)
        return updated_list

    

    def make_list_of_dicts_out_of_list_of_lists(self,list,key, one_sitting=False) -> list:
        """
        Method for changing list of lists into list of dictionaries

        Parameters:
        list(list): The list of lists to make it from
        key (str): The key for the dict
        """
        list_of_dicts = []
        if isinstance(list,dict):
            return {key:[list]}
        if list is None:
            list = [list]
        if one_sitting == True:
            return {key:list}
        for element in list:
            list_of_dicts.append({key:element})
        return list_of_dicts
    
    def give_dicts_key(self,list_of_dicts:list,key:str):
        """
        Method for giving each value a key in a dictionary

        Parameters:
        list_of_dicts (list): The list of dictionaries which need key
        key (str): The key for the value.
        """
        return [{key: value} for value in list_of_dicts]
    
    def flatten_output(self,list:list):
        """
        Method for getting rid of too many square brackets

        Parameters:
        list(list): The list with too many square brackets
        """
        flattened_list = []
        #if lists inside list have only one
        for sublist in list:
            if len(sublist) == 1:
                flattened_list.append(sublist[0])
            else:
                flattened_list.append(sublist)
        #if outer list has only one element make it only one dimension
        if len(list) == 1:
            return flattened_list[0]
        return flattened_list
    

    def remove_duplicates_in_list_of_dicts(self,input_list:list)->list:
        """
        Method for removing duplicates from a list of dicts

        input_list:list that needs its elements to be checked
        """
        unique_list_of_dicts = []
        for d in input_list:
            if d not in unique_list_of_dicts:
                unique_list_of_dicts.append(d)
        return unique_list_of_dicts

    
    def change_dict_key_name(self,dictionary,new_name,old_name):
        """
        Method for changing dictionary key name
        
        Parameters:
        dictionary: which dictionary to change it from
        new_name: what is the new name of the dicitonary
        old_name: what was the old name of the dictionary
        """
        dictionary[new_name] = dictionary.pop(old_name)

    def combine_dicts(self,dict1,dict2) -> dict:
        """
        Method for combining 2 dictionaries into one
        
        Parameters:
        dict1: first one
        dict2: second one
        """
        return {**dict1,**dict2}
    

    def check_duplicate_sitting_voting_number(self,value_list):
        """
        Method for checking duplicates of voting number in a sitting

        Parameters:
        value_list: list of lists of values to be checked 
        """
        #if value is does not exist or if there is only one value of sitting voting
        if value_list is None or isinstance(value_list,dict):
            return
        for item in value_list:
            self.check_duplicate_sitting_number(item)
    
    def check_duplicate_sitting_number(self,value_list):
        """
        Method for checking if sitting number has no duplicates
        
        Parameters:
        value_list(list): list to be searched through
        """
        used = []
        if isinstance(value_list,dict):
            return
        for item in value_list:
            if item['cislo'] in used:
                print(f"WARNING -- number of sitting (\"cislo\") -- {item['cislo']} -- is used multiple times", file=sys.stderr)
                return
            else:
                used.append(item['cislo'])

    def wrap_list_of_dicts_into_list_of_lists(self,data_list_of_dicts,key) -> list:
        """
        Wraps each dictionary inside a list with a specified key.
        
        Parameters:
        data_list (list): The list of dictionaries to wrap.
        key (str): The key for the list.
        """
        return [{key: [data]} for data in data_list_of_dicts]
    
    def wrap_leaves_into_list(self,lst, key)->list:
        """
        Method for recursive wrapping leaf nodes into a list
        """
        new_list = []
        #there is only one element in a list
        if isinstance(lst,dict):
            return {key:[lst]}
        for item in lst:
            if isinstance(item, list):
                new_list.append(self.wrap_leaves_into_list(item, key))
            elif isinstance(item, dict):
                new_item = {key: [item]}
                new_list.append(new_item)
            else:
                new_item = {key: [item]}
                new_list.append(new_item)
        return new_list
    

    def count_result_of_all_votings(self):
        """
        Method for counting if voting went through
        """
        if self.metafile_data['zastupitelstva'][0]['zasedani']['hlasovani'].get('prijato') is not None:
            return
        for council in self.output_data['zastupitelstva'] :
            for sitting in council['zasedani']:
                if council.get('zasedani') is None:
                    continue
                for voting in sitting['hlasovani']:
                    voting['prijato'] = self.count_result_of_concrete_voting(voting)

    def count_result_of_concrete_voting(self,voting) -> bool:
        """
        Method for counting the result of a concrete voting
        """
        votes_for = 0
        total_votes = 0
        for votes in voting['zastupiteleHlasy']:
            total_votes += 1
            if votes['hlas'] == "A":
                votes_for += 1
        if float(votes_for) > total_votes / 2.0:
            return True
        return False
        
    def get_validity_of_all_votings(self):
        """
        Method for asigning validity to all votings, this assumes every vote was valid
        """
        if self.metafile_data['zastupitelstva'][0]['zasedani']['hlasovani'].get('platne') is not None:
            return
        for council in self.output_data['zastupitelstva']:
            if council.get('zasedani') is None:
                continue
            if isinstance(council['zasedani'],dict):
                if self.filename.find('neplatne') == -1:
                    council['zasedani']['platne'] = True
                else:
                    council['zasedani']['platne'] = False
                return
            for sitting in council['zasedani']:
                if isinstance(sitting['hlasovani'],dict):
                    if self.filename.find('neplatne') == -1:
                        sitting['hlasovani']['platne'] = True
                    else:
                        sitting['hlasovani']['platne'] = False
                    return
                for voting in sitting['hlasovani']:
                    if self.filename.find('neplatne') == -1:
                        voting['platne'] = True
                    else:
                        voting['platne'] = False

    def give_voting_number_to_all_votings(self):
        """
        Method for giving voting number to all votings starting from one
        """
        if self.metafile_data['zastupitelstva'][0]['zasedani']['hlasovani'].get('cislo') is not None:
            return
        for council in self.output_data['zastupitelstva']:
            if council.get('zasedani') is None:
                continue
            for sitting in council['zasedani']:
                for i, voting in enumerate(sitting['hlasovani']):
                    voting['cislo'] = i + 1

    def give_sitting_number_to_all_sittings(self):
        """
        Method for giving voting number to all votings starting from one
        """
        for council in self.output_data['zastupitelstva']:
            if council.get('zasedani') is None:
                continue
            for i, sitting in enumerate(council['zasedani']):
                sitting['cislo'] = i + 1


    def split_output_by_date(self):
        """
        Method for splitting 'zasedani' by date defined in a metafile
        """
        if self.metafile_data['zastupitelstva'][0].get('_split_date') is None:
            if self.output_data['zastupitele'][0]['politickeSubjekty'][0].get('poradiZastupitelstva') is not None:
                return self.output_data
            else:
                for deputy in self.output_data['zastupitele']:
                    for party in deputy['politickeSubjekty']:
                        party['poradiZastupitelstva'] = self.output_data['zastupitelstva'][0]['poradiZastupitelstva']
        
        

        split_dates = [date['_split_date'] for date in self.metafile_data['zastupitelstva'] if date.get('_split_date') is not None]
        split_dates = [datetime.strptime(date, '%Y-%m-%d') for date in split_dates]
        split_lists = []
        copied_output = self.output_data['zastupitelstva'][0]['zasedani'].copy() #assume data are not splitted into more parts yet
        for split_date in split_dates:
            split_list = [entry for entry in copied_output if datetime.strptime(entry['datum'], '%Y-%m-%d') <= split_date] # list of part until split date
            split_lists.append(split_list)
            copied_output = [entry for entry in copied_output if datetime.strptime(entry['datum'], '%Y-%m-%d') > split_date] # list of the rest of council
        split_lists.append(copied_output)

        for i, one_part in enumerate(split_lists):
            self.output_data['zastupitelstva'][i]['zasedani'] = []
            self.output_data['zastupitelstva'][i]['zasedani'] = one_part


        council_order_list = []
        for council in self.output_data['zastupitelstva']:
            council_order_list.append(council['poradiZastupitelstva'])


        if self.metafile_data['zastupitele'].get('politickeSubjekty') is not None:
            return
        
        last_first_voting_date = []
        # for correct output end date and start dates of party partinence
        for index, element in enumerate(self.output_data['zastupitelstva']):
            if element != self.output_data['zastupitelstva'][-1]:
                last_first_voting_date.append((element['zasedani'][-1]['datum'],self.output_data['zastupitelstva'][index+1]['zasedani'][0]['datum']))
                
        #deputies political subjects by split date
        copied_output = self.output_data['zastupitele'].copy()
        for index, split_date in enumerate(split_dates):
            for deputy in copied_output:
                new_parties_list = []
                for party in deputy["politickeSubjekty"]:
                    if datetime.strptime(party['do'], '%Y-%m-%d') > split_date and datetime.strptime(party['od'], '%Y-%m-%d') < split_date:
                        new_parties_list.append({'od':party['od'],
                                                 "do":last_first_voting_date[index][0],
                                                 "idPolitickySubjekt":party['idPolitickySubjekt'],
                                                 "poradiZastupitelstva":council_order_list[index]})
                        new_parties_list.append({'od':last_first_voting_date[index][1],
                                                 "do":party['do'],
                                                 "idPolitickySubjekt":party['idPolitickySubjekt'],
                                                 "poradiZastupitelstva":council_order_list[index+1]})
                    else:
                        if datetime.strptime(party['od'], '%Y-%m-%d') > split_date:
                            new_parties_list.append({'od':party['od'],
                                                    "do":party['do'],
                                                    "idPolitickySubjekt":party['idPolitickySubjekt'],
                                                    "poradiZastupitelstva":council_order_list[index+1]})
                        else:
                            new_parties_list.append({'od':party['od'],
                                                 "do":party['do'],
                                                 "idPolitickySubjekt":party['idPolitickySubjekt'],
                                                 "poradiZastupitelstva":council_order_list[index]})
                    deputy['politickeSubjekty'] = new_parties_list
        return

    
