import sys
import re
from datetime import datetime
from src.Parser.TreeFileParser import TreeFileParser



class DataClean:
    """Class for data cleaning"""
    def __init__(self,data,metafile) -> None:
        self.metafile = metafile
        self.data = data

    def clean_data(self):
        """
        Method for cleaning data
        """
        if self.metafile['config']['_cleaning'] is None:
            return self.data
        for element in self.metafile['config']['_cleaning']:
            self.clean_concrete_data(element)
        print("INFO: Finished cleaning data")
        return self.data
    def clean_concrete_data(self,info):
        """
        Method for cleaning concrete data

        Parameters:
            info: concrete info about data to be cleaned
        """
        match info['method']:
            case "CombineCellsAndDeleteSecond":
                self.combine_cells_and_delete_second(info['parameters'])
            case "CombineCellsAndKeepBoth":
                self.combine_cells_and_keep_both(info['parameters'])
            case "DeleteRow":
                self.delete_row(info['parameters'])
            case "ChangeValueInTree":
                self.change_value_in_tree(info['parameters'],"ChangeValueInTree")
            case "ChangeDeputyPartyValueInDateInterval":
                self.change_value_in_tree(info['parameters'],"ChangeDeputyPartyValueInDateInterval")
            case _:
                print(f"{info['method']} is not known cleaning type",file=sys.stderr)

    def combine_cells_and_delete_second(self,parameters):
        """
        Method for combining data cells in csv 

        Parameters:
            parameters: parameters needed for succesful combining of cells
        """
        y1,y2 = parameters['first_cell_y'],parameters['second_cell_y']
        x1,x2 = parameters['first_cell_x'],parameters['second_cell_x']
        if (0 <= y1 < len(self.data)) and (0 <= y2 < len(self.data)) and (0 <= x1 < len(self.data[y1])) and (0 <= x2 < len(self.data[y2])):
            #combine elements at the specified positions
            self.data[y1][x1] = self.data[y1][x1] + self.data[y2][x2]
            del self.data[y2][x2]
            return
        print("ERROR: Wrong position of Combine cells provided while cleaning data",file=sys.stderr)
    
    def combine_cells_and_keep_both(self,parameters):
        """
        Method for combining data cells in csv and keeping the second value as before

        Parameters:
            parameters: parameters needed for succesful combining of cells
        """
        y1,y2 = parameters['first_cell_y'],parameters['second_cell_y']
        x1,x2 = parameters['first_cell_x'],parameters['second_cell_x']

        if (0 <= y1 < len(self.data)) and (0 <= y2 < len(self.data)) and (0 <= x1 < len(self.data[y1])) and (0 <= x2 < len(self.data[y2])):
            #combine elements at the specified positions
            self.data[y1][x1] = self.data[y1][x1][:(parameters.get('_first_cell_cut') if parameters.get('_first_cell_cut') is not None else len(self.data[y1][x1]))] + parameters['cell_divider'] + self.data[y2][x2][:(parameters.get('_second_cell_cut') if parameters.get('_second_cell_cut') is not None else len(self.data[y2][x2]))]
            return
        print("ERROR: Wrong position of Combine cells provided while cleaning data",file=sys.stderr)


    def delete_row(self,parameters):
        """
        Method for deleting row by parameters
        """
        row_index = parameters['row']
        print(f"INFO: You deleted row: {self.data[row_index]}",file=sys.stderr)
        print(f"If it is a differenct row than you expected it may be because of deleting previous data so the index shifts",file=sys.stderr)
        del self.data[row_index]

    def change_value_in_tree(self,parameters,clean_type):
        """Method for changing value to a different one in whole tree file"""
        tree_pos = parameters['tree_position']
        self.change_data_in_tree(tree_pos,
                                   'politickeSubjekty',
                                   clean_type,
                                   parameters)


    def change_data_in_tree_with_certain_values(self, data, list_of_tree_parts,current_date,value_type=None, value_params=None ):
        """
        Method that recursively changes data in a tree file, ispired by recursively search tree method
        """
        tree_file = TreeFileParser(self.metafile,"")
        if list_of_tree_parts[0][1] == ".":
            
            for element in data[list_of_tree_parts[0][0]]:
                self.change_data_in_tree_with_certain_values(element, list_of_tree_parts[1:],current_date, value_type, value_params)
        elif list_of_tree_parts[0][1] == "":
            data[list_of_tree_parts[0][0]] = tree_file.convert_string_to_list(data[list_of_tree_parts[0][0]])
            data[list_of_tree_parts[0][0]] = tree_file.apply_type_rule_on_list(data[list_of_tree_parts[0][0]], "ChangeValueInTree", [data[list_of_tree_parts[0][0]][0],value_params['new_value']])
        elif list_of_tree_parts[0][1] == "|":
            list_of_parts = []
            for part in list_of_tree_parts:
                if data.get(part[0]) is not None:
                    one_part = tree_file.convert_string_to_list(data[part[0]])
                    one_part = tree_file.apply_type_rule_on_list(one_part, value_type, value_params)
                    list_of_parts = list_of_parts + one_part
        elif list_of_tree_parts[0][1] == "`":
            self.change_data_in_tree_with_certain_values(data[list_of_tree_parts[0][0]], list_of_tree_parts[1:],current_date, value_type, value_params)
        elif list_of_tree_parts[0][1] == "*":
            parts = re.split(r'(\[.*?\])', list_of_tree_parts[0][0])
            
            if parts[1] == "[date]":
                element = tree_file.convert_string_to_list(data[parts[0]]).copy()
                element = tree_file.apply_type_rule_on_list(element, "DateStringType", [value_params['original_date_format']])
                if value_params.get('start_date_limit') is None:
                    value_params['start_date_limit'] = "1900-01-01"
                if value_params.get('end_date_limit') is None:
                    value_params['end_date_limit'] = "9999-12-31"
                
                if datetime.strptime(value_params['start_date_limit'], '%Y-%m-%d') <= datetime.strptime(element[0], '%Y-%m-%d') <= datetime.strptime(value_params['end_date_limit'], '%Y-%m-%d'):
                    self.change_data_in_tree_with_certain_values(data, list_of_tree_parts[1:],current_date, value_type, value_params)
            if parts[1] == "[name]":
                element = tree_file.convert_string_to_list(data[parts[0]]).copy()
                if value_params['name'] == element[0]:
                    self.change_data_in_tree_with_certain_values(data, list_of_tree_parts[1:],current_date, value_type, value_params)
            current_date = []
        else:
            print(f"Check your metafile -- unknown delimiter (viable are: \".\",\"|\",\"`\",\"+\",,\"*\")", file=sys.stderr)



    def change_data_in_tree(self, tree_position:str,key:str,value_type=None,value_params=None):
        """
        Method that changes data that is written in a tree
        """
        tree_file_parser = TreeFileParser(self.metafile,"")
        list_of_tree_parts = tree_file_parser.split_string_by_viable_split_chars(tree_position)
        if value_type == "ChangeValueInTree":
            self.recursively_change_data_in_tree(self.data,list_of_tree_parts,value_type,[value_params['old_name'], value_params['new_name']])
        elif value_type == "ChangeDeputyPartyValueInDateInterval":
            self.change_data_in_tree_with_certain_values(self.data,list_of_tree_parts,[],value_type,value_params)
        return
    

    def recursively_change_data_in_tree(self, data, list_of_tree_parts, value_type=None, value_params=None):
        """
        Method that recursively changes data in a tree file, ispired by recursively search tree method
        """
        tree_file = TreeFileParser(self.metafile,"")
        if list_of_tree_parts[0][1] == ".":
            for element in data[list_of_tree_parts[0][0]]:
                self.recursively_change_data_in_tree(element, list_of_tree_parts[1:], value_type, value_params)
        elif list_of_tree_parts[0][1] == "":
            data[list_of_tree_parts[0][0]] = tree_file.convert_string_to_list(data[list_of_tree_parts[0][0]])
            data[list_of_tree_parts[0][0]] = tree_file.apply_type_rule_on_list(data[list_of_tree_parts[0][0]], value_type, value_params)
        elif list_of_tree_parts[0][1] == "|":
            list_of_parts = []
            for part in list_of_tree_parts:
                if data.get(part[0]) is not None:
                    one_part = tree_file.convert_string_to_list(data[part[0]])
                    one_part = tree_file.apply_type_rule_on_list(one_part, value_type, value_params)
                    list_of_parts = list_of_parts + one_part
        elif list_of_tree_parts[0][1] == "`":
            self.recursively_change_data_in_tree(data[list_of_tree_parts[0][0]], list_of_tree_parts[1:], value_type, value_params)
        else:
            print(f"Check your metafile -- unknown delimiter (viable are: \".\",\"|\",\"`\")", file=sys.stderr)

